import logging
import time

from sandbox import sdk2
from sandbox.projects.browser.common.binary_tasks import LinuxBinaryTaskMixin
from sandbox.projects.browser.common.contextmanagers import ExitStack, TempEnvironment
from sandbox.projects.browser.common.git import repositories
from sandbox.projects.browser.metrics.common import BrowserMeasureGitTime

import six
if six.PY2:
    import pathlib2 as pathlib
else:
    import pathlib


PUSH_CHART_FEED_ID = '8x9h5s73vtv40'


class BrowserMeasureGitPushTime(BrowserMeasureGitTime, LinuxBinaryTaskMixin, sdk2.Task):
    class Parameters(BrowserMeasureGitTime.Parameters):
        base_branch = sdk2.parameters.String(
            'Base branch (e.g. \'metrics/push/basic\', \'metrics-fetch-from-branch\').'
            ' New branch is created from this branch HEAD',
            required=True
        )

        folder_to_change_files = sdk2.parameters.String(
            'Relative path inside repo (e.g. \'yin/sheriff/bitbucket_prs\', \'src/tools/polymer\').'
            ' All files inside this path (and its subdirectories) will be changed ONLY for temporary commit',
            required=True
        )

    @property
    def branch_to_push(self):
        # TODO (BYIN-14493): switch to .timestamp() after dropping python2
        # TODO (BYIN-14493): {time} MUST be int not float
        return 'metrics/push/temporary_{time}'.format(time=int(self.created_with_timezone_timestamp))

    @property
    def yt_table_path_template(self):
        return '//home/browser/infra/statistics/bitbucket_git/push/{date}'

    def get_push_time_seconds(self, branch_to_push, transfer_protocol, ssh_key_optional):
        """
        :param branch_to_push: name of branch to push changes
        :type branch_to_push: str
        :param transfer_protocol: either 'https' or 'ssh'
        :type transfer_protocol: str
        :param ssh_key_optional: pass None if no SSH involved
        :type ssh_key_optional: sdk2.ssh.Key|None
        :rtype: float
        """
        logging.debug('git log: %s', self.git_cli.log('-n', '1'))

        # Create branch named 'branch_to_push' from base_branch.
        self.git_cli.checkout('-b', branch_to_push)

        # Change all files inside self.Parameters.folder_to_change_files.
        for full_path in pathlib.Path(
            self.local_repo_absolute_path, self.Parameters.folder_to_change_files
        ).glob('**/*'):
            if full_path.is_dir():
                continue
            # Reverse all line inside file at full_path.
            with full_path.open('r', encoding='utf-8') as f:
                file_lines = f.readlines()
            with full_path.open('w', encoding='utf-8') as f:
                for line in reversed(file_lines):
                    f.write(line)

            # Add file to commit.
            self.git_cli.add(str(full_path))

        # Do the commit.
        self.git_cli.commit('-m', 'temporary test commit for BROWSER_MEASURE_GIT_PUSH_TIME')
        logging.debug('git log: %s', self.git_cli.log('-n', '1'))

        with ExitStack() as stack:
            # Do the push.
            if transfer_protocol == 'ssh':
                stack.enter_context(ssh_key_optional)

            time_before_push = time.time()

            with TempEnvironment() as temp_env:
                temp_env.set_var('GIT_TRACE', '1')
                self.git_cli.push('--set-upstream', 'origin', branch_to_push)

            time_after_push = time.time()

            # Delete the new branch remotely.
            # Local branch will be deleted with the entire folder while self.clear_local_repo().
            self.git_cli.push('-d', 'origin', branch_to_push)

        time_taken_in_seconds = time_after_push - time_before_push
        logging.debug('Took %.3f seconds', time_taken_in_seconds)
        return time_taken_in_seconds

    @property
    def yt_table_attributes_json(self):
        """
        :return: JSON with scheme for data that's going to be uploaded to YT
        :rtype: dict[str, Any]
        """
        return {
            'unique_keys': True,
            'schema': {
                '$value': [
                    {
                        'type': 'float',
                        'name': 'execute_timestamp',
                        'required': True
                    },
                    # Branch/commit information
                    {
                        'type': 'string',
                        'name': 'base_branch',
                        'required': True
                    },
                    {
                        'type': 'string',
                        'name': 'base_branch_commit',
                        'required': True
                    },
                    {
                        'type': 'string',
                        'name': 'folder_to_change_files',
                        'required': True
                    },
                    # Launch parameters
                    {
                        'type': 'string',
                        'name': 'project_id',
                        'required': True
                    },
                    {
                        'type': 'string',
                        'name': 'repository_id',
                        'required': True
                    },
                    {
                        'type': 'string',
                        'name': 'transfer_protocol',
                        'required': True
                    },
                    # Result
                    {
                        'type': 'float',
                        'name': 'push_time_seconds',
                        'required': True
                    },
                ],
                '$attributes': {
                    'strict': True,
                }
            }
        }

    def build_statistics_for_yt(self, push_time_seconds, execute_timestamp, branch, commit, folder_to_change_files):
        """
        :param push_time_seconds: time taken for git-push
        :type push_time_seconds: float
        :type execute_timestamp: float
        :type branch: str
        :type commit: str
        :type folder_to_change_files: str
        :rtype: dict[str, Any]
        """
        return {
            'execute_timestamp': execute_timestamp,

            'base_branch': branch,
            'base_branch_commit': commit,
            'folder_to_change_files': folder_to_change_files,

            'project_id': self.PROJECT_ID,
            'repository_id': self.Parameters.repository_id,
            'transfer_protocol': self.Parameters.transfer_protocol,

            'push_time_seconds': push_time_seconds,
        }

    def prepare_local_repo(self):
        """
        :rtype: None
        """
        # fetch-url is always HTTPS, push-url depends on self.Parameters.transfer_protocol.
        if self.Parameters.transfer_protocol == 'https':
            token = self.Parameters.bitbucket_yt_token_secret.data()[
                self.Parameters.bitbucket_yt_token_secret.default_key
            ]
            push_url = repositories.get_https_git_url(
                repo=self.Parameters.repository_id,
                project=self.PROJECT_ID,
                login=self.ROBOT_LOGIN,
                token=token
            )
        elif self.Parameters.transfer_protocol == 'ssh':
            push_url = repositories.get_ssh_git_url(
                repo=self.Parameters.repository_id,
                project=self.PROJECT_ID
            )
        else:
            assert False, 'Not reachable'

        # For git-push measuring, cacheable Git can and should be used (unlike for git-fetch measuring).
        repository = repositories.Git(
            url=repositories.get_https_git_url(
                repo=self.Parameters.repository_id,
                project=self.PROJECT_ID
            ),
            push_url=push_url
        )
        repository.clone(str(self.local_repo_absolute_path), self.Parameters.base_branch)

    def get_measured_git_commands(self):
        # Git-push branch_to_push, measure time of git-push executing.
        push_time_seconds = self.get_push_time_seconds(
            branch_to_push=self.branch_to_push,
            ssh_key_optional=self.ssh_key_optional,
            transfer_protocol=self.Parameters.transfer_protocol
        )

        # Get base branch's HEAD commit id.
        base_branch_commit = self.get_commit_by_branch(
            branch=self.Parameters.base_branch,
            ssh_key_optional=self.ssh_key_optional,
            transfer_protocol=self.Parameters.transfer_protocol
        )

        push_statistics = self.build_statistics_for_yt(
            push_time_seconds=push_time_seconds,
            # TODO (BYIN-14493): switch to .timestamp() after dropping python2
            execute_timestamp=self.created_with_timezone_timestamp,
            branch=self.Parameters.base_branch,
            commit=base_branch_commit,
            folder_to_change_files=self.Parameters.folder_to_change_files
        )

        return [push_statistics]

    @property
    def feed_ids(self):
        """
        :return: feed IDs of corresponding charts
        :rtype: list[str]
        """
        return [
            PUSH_CHART_FEED_ID,
        ]
