import abc
from datetime import datetime
import logging
import pytz
import shutil
import time

import sandbox
from sandbox import sdk2
from sandbox.common.errors import TemporaryError
import sandbox.common.types.client as ctc
from sandbox.projects.browser.common.binary_tasks import DEFAULT_LXC_CONTAINER_RESOURCE_ID
from sandbox.projects.browser.common.contextmanagers import ExitStack
from sandbox.projects.browser.common.git import GitEnvironment
from sandbox.projects.browser.common.git.git_cli import GitCli
from sandbox.projects.common import decorators
from sandbox.sdk2.helpers import subprocess

PREPARATION_FAILED_COMMENT_COLOR = '#ff8000'
DATA_COLLECTION_FAILED_COMMENT_COLOR = '#ff0000'
DATA_REPORTING_FAILED_COMMENT_COLOR = '#ff00ff'


class BrowserMeasureGitTime:
    LOCAL_REPO_RELATIVE_PATH = 'browser-measure-git-repo'
    PROJECT_ID = 'stardust'
    ROBOT_LOGIN = 'robot-browser-infra'

    class Requirements(sdk2.Task.Requirements):
        container_resource = DEFAULT_LXC_CONTAINER_RESOURCE_ID
        client_tags = ctc.Tag.BROWSER
        environments = (
            GitEnvironment('2.32.0'),
        )
        cores = 2
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass  # Do not use any shared caches (required for running on multislot agent)

    class Parameters(sdk2.Parameters):
        fail_on_any_error = True

        # Default value is a secret of robot-bro-sheriff.
        # This secret has a token for app browser-metrics-bitbucket-git, which has access to Bitbucket and YT API.
        bitbucket_yt_token_secret = sdk2.parameters.YavSecret(
            'YAV secret identifier for access to Bitbucket API and YT API',
            default='sec-01cn4cz7j3tryd6n540b5ebgkj#browser-metrics-bitbucket-git-token'
        )

        with sdk2.parameters.Group('Repository settings') as repository_settings:
            with sdk2.parameters.RadioGroup('ID of repository', required=True) as repository_id:
                repository_id.values['browser'] = repository_id.Value('browser', default=True)
                repository_id.values['yin'] = repository_id.Value('yin')

            with sdk2.parameters.RadioGroup('Transfer protocol', required=True) as transfer_protocol:
                transfer_protocol.values['https'] = transfer_protocol.Value('HTTPS')
                transfer_protocol.values['ssh'] = transfer_protocol.Value('SSH')

            with transfer_protocol.value['ssh']:
                # Default value is a secret of robot-bro-sheriff.
                # This secret has a private part of SSH key for git commands interacting with Bitbucket.
                private_ssh_key_secret = sdk2.parameters.YavSecret(
                    'YAV secret identifier for SSH access to Bitbucket',
                    default='sec-01cn4cz7j3tryd6n540b5ebgkj#browser-metrics-bitbucket-git-ssh-key',
                    required=True
                )

        dry_run = sdk2.parameters.Bool('dry-run: do not upload data to YT', default=False)

    @decorators.memoized_property
    def local_repo_absolute_path(self):
        """
        :rtype: pathlib.Path
        """
        # self.path is taken from sdk2.Task
        return self.path(self.LOCAL_REPO_RELATIVE_PATH).absolute()

    @sandbox.common.utils.singleton_property
    def git_cli(self):
        return GitCli(
            str(self.local_repo_absolute_path),
            config={
                'user.name': self.ROBOT_LOGIN,
                'user.email': '{}@yandex-team.ru'.format(self.ROBOT_LOGIN),
            }
        )

    def get_commit_by_branch(self, branch, ssh_key_optional, transfer_protocol):
        """
        :type branch: str
        :param ssh_key_optional: pass None if no SSH involved
        :type ssh_key_optional: sdk2.ssh.Key|None
        :param transfer_protocol: either 'https' or 'ssh'
        :type transfer_protocol: str
        :return: commit id
        :rtype: str
        """
        assert transfer_protocol != 'ssh' or ssh_key_optional is not None, 'Specify SSH key or do not use SSH'

        with ExitStack() as stack:
            if transfer_protocol == 'ssh':
                stack.enter_context(ssh_key_optional)
            ls_remote_output = self.git_cli.ls_remote('origin', branch)
        commit_id, _ = ls_remote_output.decode().strip().split('\t', 1)
        return commit_id

    @property
    @abc.abstractmethod
    def yt_table_path_template(self):
        """
        Return str with one placeholder named 'date' to format execute_date into it.
        For example: 'some_template_{date}'

        :rtype: str
        """
        pass

    def get_yt_table_path(self, execute_date):
        """
        :type execute_date: str
        :rtype: str
        """
        return self.yt_table_path_template.format(date=execute_date)

    @property
    @sandbox.common.utils.singleton
    def yt_client(self):
        import yt.wrapper as yt

        token = self.Parameters.bitbucket_yt_token_secret.data()[self.Parameters.bitbucket_yt_token_secret.default_key]
        return yt.YtClient(proxy='hahn', token=token)

    def upload_to_yt(self, statistics, execute_date):
        """
        :param statistics: list where every item stores result of one git command
        according to scheme from self.yt_table_attributes_json
        :type statistics: list[dict[str, Any]]
        :type execute_date: str
        :rtype: None
        """
        import yt.wrapper as yt
        import yt.yson as yson

        yt_table_path = self.get_yt_table_path(execute_date)
        logging.debug('yt_table_path == %s', yt_table_path)

        yt_table = yt.TablePath(yt_table_path, append=True)

        if not yt.exists(yt_table_path, client=self.yt_client):
            yt.create(
                'table',
                yt_table_path,
                attributes=yson.json_to_yson(self.yt_table_attributes_json),
                client=self.yt_client
            )
        yt.write_table(
            yt_table, statistics, format=yt.JsonFormat(attributes={'encode_utf8': True}), client=self.yt_client
        )

    @property
    @abc.abstractmethod
    def yt_table_attributes_json(self):
        """
        :return: JSON with scheme for data that's going to be uploaded to YT
        :rtype: dict[str, Any]
        """
        pass

    @abc.abstractmethod
    def build_statistics_for_yt(self, *args, **kwargs):
        """
        :rtype: dict[str, Any]
        """
        pass

    @staticmethod
    def clear_local_repo(local_repo_absolute_path):
        """
        :type local_repo_absolute_path: pathlib.Path
        """
        shutil.rmtree(str(local_repo_absolute_path))

    @decorators.memoized_property
    def ssh_key_optional(self):
        """
        Return ssh key if it's needed. Else return None.

        :rtype: sdk2.ssh.Key|None
        """
        ssh_key_optional = None
        if self.Parameters.transfer_protocol == 'ssh':
            ssh_key_optional = sdk2.ssh.Key(
                private_part=self.Parameters.private_ssh_key_secret.data()[
                    self.Parameters.private_ssh_key_secret.default_key
                ]
            )
        return ssh_key_optional

    @abc.abstractmethod
    def prepare_local_repo(self):
        """
        :rtype: None
        """
        pass

    @abc.abstractmethod
    def get_measured_git_commands(self):
        """
        Return statistics in form suitable to upload to YT.

        :rtype: list[dict[str, Any]]
        """
        pass

    @decorators.memoized_property
    def created_with_timezone(self):
        local_timezone = pytz.timezone("Europe/Moscow")
        return self.created.astimezone(tz=local_timezone)

    @decorators.memoized_property
    def created_with_timezone_timestamp(self):
        """
        :rtype: float
        """
        return time.mktime(self.created_with_timezone.timetuple())

    @property
    @abc.abstractmethod
    def feed_ids(self):
        """
        :return: feed IDs of corresponding charts
        :rtype: list[str]
        """
        pass

    @staticmethod
    def add_comment_to_charts(feed_ids, event_timestamp, note_text, comment_color, token):
        import library.python.charts_notes as notes

        datetime_iso = datetime.fromtimestamp(event_timestamp).isoformat()
        for feed_id in feed_ids:
            logging.debug('Add comment to feed id #%s, date %s', feed_id, datetime_iso)
            notes.create(
                feed=feed_id,
                date=datetime_iso,
                note=notes.Line(
                    text=note_text,
                    color=comment_color,
                ),
                oauth_token=token
            )

    def on_execute(self):
        logging.info('Task started to execute')

        oauth_token=self.Parameters.bitbucket_yt_token_secret.data()[
            self.Parameters.bitbucket_yt_token_secret.default_key
        ]

        # Prepare a Git repository.
        try:
            self.prepare_local_repo()
        except TemporaryError:
            logging.debug('Preparation failed. Add a comment to charts')
            self.add_comment_to_charts(
                feed_ids=self.feed_ids,
                event_timestamp=self.created_with_timezone_timestamp,
                note_text='Preparation failed #{}'.format(self.id),
                comment_color=PREPARATION_FAILED_COMMENT_COLOR,
                token=oauth_token
            )
            logging.debug('Comment added')
            raise

        # Get statistics in format suitable to upload to YT.
        try:
            statistics = self.get_measured_git_commands()
        except subprocess.CalledProcessError:
            logging.debug('Data collection failed. Add a comment to charts')
            self.add_comment_to_charts(
                feed_ids=self.feed_ids,
                event_timestamp=self.created_with_timezone_timestamp,
                note_text='Data collection failed #{}'.format(self.id),
                comment_color=DATA_COLLECTION_FAILED_COMMENT_COLOR,
                token=oauth_token
            )
            logging.debug('Comment added')
            raise

        # Do the cleanup.
        self.clear_local_repo(self.local_repo_absolute_path)

        if self.Parameters.dry_run:
            logging.debug('dry-run')
            logging.debug('statistics: %s', statistics)
            return

        # execute_date is a part of YT table path. New table is created once per day.
        # self.created is taken from sdk2.Task
        execute_date = self.created_with_timezone.strftime('%Y-%m-%d')

        # Upload data to YT.
        try:
            self.upload_to_yt(statistics=statistics, execute_date=execute_date)
        except Exception:
            logging.debug('Data reporting failed. Add a comment to charts')
            self.add_comment_to_charts(
                feed_ids=self.feed_ids,
                event_timestamp=self.created_with_timezone_timestamp,
                note_text='Data reporting failed #{}'.format(self.id),
                comment_color=DATA_REPORTING_FAILED_COMMENT_COLOR,
                token=oauth_token
            )
            logging.debug('Comment added')
            raise
