# -*- encoding: utf-8 -*-

import calendar
import contextlib
import datetime
import logging
import os
import time

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.common.utils import singleton_classproperty
from sandbox.projects.browser.monitoring import config
from sandbox.projects.browser.monitoring import processor
from sandbox.projects.common import solomon
from sandbox.sandboxsdk.environments import PipEnvironment


class Config(sdk2.parameters.String):
    @singleton_classproperty
    def default_value(cls):
        config_names = config.MonitorConfigLoader.available_configs().keys()
        return config_names[0] if config_names else '---'


class BrowserMonitoringLogsAnalyser(sdk2.Task):
    """
    Requests values from Metrika mobile log through YQL and sends them to Solomon.
    """

    _SECRET_KEY_SOLOMON_TOKEN = 'solomon-token'
    _SECRET_KEY_YQL_TOKEN = 'yql-token'
    _SOLOMON_MAX_SENSORS_TO_SEND = 10000

    class Requirements(sdk2.Task.Requirements):
        disk_space = 1024
        cores = 1
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.BROWSER

        environments = (
            PipEnvironment('jsonschema', version='2.6.0'),
            # 'yql' depends on 'yandex-yt' but does not install it.
            PipEnvironment('yandex-yt', '0.9.35'),
            PipEnvironment('yql', '1.2.97'),
        )

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = datetime.timedelta(hours=1).total_seconds()

        secrets_uuid = sdk2.parameters.YavSecret(
            'secrets', description='YAV UUID of secrets.',
            default='sec-01en00fygjhjz9apd8knr1rhng')

        static_pool_name = sdk2.parameters.String(
            'YQL Static Pool', description='Name of YQL static pool to use '
                                           '(Default pool if not set).',
            default='')

        timestamp = sdk2.parameters.Integer(
            'timestamp', description='Query timestamp in UTC.',
            default=int(calendar.timegm(datetime.datetime.utcnow().timetuple())))
        delta = sdk2.parameters.Integer(
            'delta', description='Time step in seconds.',
            default=int(datetime.timedelta(minutes=30).total_seconds()))
        count = sdk2.parameters.Integer(
            'count', description='Step count.',
            default=1)
        refresh_yql_cache = sdk2.parameters.Bool(
            'Refresh', description='Refresh YQL cache.',
            default=False)

        with Config('Config') as config_name:
            for choice_name in config.MonitorConfigLoader.available_configs().keys():
                config_name.values[choice_name] = choice_name

        with sdk2.parameters.Group('Solomon') as sandbox_lite_settings:
            solomon_send = sdk2.parameters.Bool(
                'Send', description='Send metrics to Solomon (or just log them)',
                default=False)
            solomon_project = sdk2.parameters.String(
                'Project', description='Solomon project. '
                                       'Leave empty to use value from config.')
            solomon_cluster = sdk2.parameters.String(
                'Cluster', description='Solomon cluster. '
                                       'Leave empty to use value from config.')
            solomon_service_raw = sdk2.parameters.String(
                'Service [raw]', description='Solomon service for raw data. '
                                             'Leave empty to use value from config.')
            solomon_service_processed = sdk2.parameters.String(
                'Service [processed]', description='Solomon service for processed data. '
                                                   'Leave empty to use value from config.')
            solomon_service_generated = sdk2.parameters.String(
                'Service [generated]', description='Solomon service for generated data. '
                                                   'Leave empty to use value from config.')

        with sdk2.parameters.Group('Retry') as retry_settings:
            retry_count = sdk2.parameters.Integer(
                'Retry count', description='Count of retries if YQL query fails.',
                default=3)
            retry_delay_secs = sdk2.parameters.Integer(
                'Retry delay', description='Delay between tries in seconds.',
                default=datetime.timedelta(hours=1).total_seconds())

    @contextlib.contextmanager
    def step(self, step_name):
        start_time = time.time()
        try:
            logging.info('[START]  %s ...', step_name)
            yield
        finally:
            total_time = time.time() - start_time
            mins = int(total_time / 60)
            secs = int(total_time - mins * 60)
            time_str_list = []
            if mins:
                time_str_list.append('{} min'.format(mins))
            time_str_list.append('{} sec'.format(secs))
            time_str = ' '.join(time_str_list)
            logging.info('[FINISH] %s in %s', step_name, time_str)

    def get_monitor_datetime_logger(self, query_datetime, prefix):
        log_name = query_datetime.strftime('%Y-%m-%d-%H-%M-%S.txt')
        if prefix:
            log_name = '{}-{}'.format(prefix, log_name)
        log_path = os.path.join(str(self.log_resource.path), log_name)

        logger = logging.getLogger(log_name)
        handler = logging.FileHandler(log_path)
        handler.setFormatter(logging.Formatter('%(message)s'))
        logger.addHandler(handler)
        logger.setLevel(logging.DEBUG)

        return logger

    def log_version_data_table(self, prefix, query_datetime, version_data_table):
        logger = self.get_monitor_datetime_logger(query_datetime, prefix)
        for point in version_data_table.points:
            report_datetime = datetime.datetime.utcfromtimestamp(point.report_timestamp)
            logger.debug('Point "%s" [%s] : %s', report_datetime, point.version_name, point.weight)
            for name in sorted(point.values.keys()):
                logger.debug(' - %s : %s', name, point.values[name])

    def log_version_labels_dict(self, prefix, query_datetime, version_labels_dict):
        logger = self.get_monitor_datetime_logger(query_datetime, prefix)
        for version_name in sorted(version_labels_dict.keys(), cmp=processor.compare_versions):
            logger.debug('Version labels of "%s": %s', version_name, version_labels_dict[version_name])

    def log_version_mapping(self, prefix, query_datetime, version_mapping):
        logger = self.get_monitor_datetime_logger(query_datetime, prefix)
        logger.debug('Version mapping: %s', ', '.join([
            '{} -> {}'.format(source_version, target_version)
            for target_version, source_version in version_mapping.iteritems()]))

    @staticmethod
    def send_sensors_to_solomon(project, cluster, service, solomon_token,
                                solomon_sensors, max_sensors_to_send=_SOLOMON_MAX_SENSORS_TO_SEND):
        while solomon_sensors:
            sensors_part = solomon_sensors[:max_sensors_to_send]
            solomon_sensors = solomon_sensors[max_sensors_to_send:]
            logging.info('Send %s sensors to Solomon ...', len(sensors_part))
            solomon.push_to_solomon_v2(
                params={'project': project, 'cluster': cluster, 'service': service},
                token=solomon_token, common_labels={}, sensors=sensors_part)

    def solomon_send(self, solomon_project, solomon_cluster, solomon_service, solomon_sensors):
        if self.Parameters.solomon_send:
            solomon_token = self.Parameters.secrets_uuid.data()[self._SECRET_KEY_SOLOMON_TOKEN]
            self.send_sensors_to_solomon(
                solomon_project,
                solomon_cluster,
                solomon_service,
                solomon_token,
                solomon_sensors)
        else:
            logging.info('Skip sending to Solomon')

    def on_execute(self):
        config_path = config.MonitorConfigLoader.available_configs()[self.Parameters.config_name]
        logging.info('monitor config path: %s', config_path)
        monitor_config = config.MonitorConfigLoader.load_config(config_path)
        logging.info('YQL fresh template path: %s', monitor_config.yql_fresh_template_path)
        logging.info('YQL stale template path: %s', monitor_config.yql_stale_template_path)

        # Initialize retry info.
        if self.Context.retry_count is ctm.NotExists:
            self.Context.start = int(self.Parameters.timestamp)
            self.Context.count = int(self.Parameters.count)
            self.Context.retry_count = int(self.Parameters.retry_count)

        timestamp_range = range(
            self.Context.start,
            self.Context.start + self.Parameters.delta * self.Context.count,
            self.Parameters.delta)
        for timestamp in timestamp_range:
            try:
                self.monitor(monitor_config, self.Parameters.static_pool_name, timestamp)

                # Update retry info.
                self.Context.start = timestamp + self.Parameters.delta
                self.Context.count -= 1
                self.Context.retry_count = int(self.Parameters.retry_count)
            except processor.MonitorError as e:
                logging.exception('monitor error [%s]: %s', e.code, e.message)

                self.Context.retry_count -= 1
                if self.Context.retry_count >= 0:
                    raise sdk2.WaitTime(self.Parameters.retry_delay_secs)
                else:
                    raise

    def monitor(self, monitor_config, static_pool_name, timestamp):
        solomon_project = self.Parameters.solomon_project or monitor_config.solomon_project
        solomon_cluster = self.Parameters.solomon_cluster or monitor_config.solomon_cluster
        solomon_service_raw = (self.Parameters.solomon_service_raw
                               or monitor_config.solomon_service_raw)
        solomon_service_processed = (self.Parameters.solomon_service_processed
                                     or monitor_config.solomon_service_processed)
        solomon_service_generated = (self.Parameters.solomon_service_generated
                                     or monitor_config.solomon_service_generated)

        monitor_query = processor.MonitorQuery(monitor_config, static_pool_name, timestamp)
        query_datetime = datetime.datetime.utcfromtimestamp(monitor_query.query_timestamp)
        logging.info('query datetime : %s', query_datetime)

        with self.step('Fetch data'):
            monitor_values = processor.MonitorValues.load(
                self.Parameters.config_name, monitor_query.query_timestamp)
            if not monitor_values or self.Parameters.refresh_yql_cache:
                yql_token = self.Parameters.secrets_uuid.data()[self._SECRET_KEY_YQL_TOKEN]
                monitor_values = processor.MonitorValues.fetch(monitor_query, yql_token)
                monitor_values.save(self, self.Parameters.config_name)

        version_data_table = processor.VersionDataManager.load_version_data_table(
            monitor_values, monitor_config.weight_sensor_name)
        wrong_versions = processor.VersionDataManager.get_wrong_versions(
            version_data_table.versions, monitor_config.version_pattern)
        logging.info('Wrong versions: %s', wrong_versions)
        version_data_table = version_data_table.build_without_versions(wrong_versions)
        self.log_version_data_table('raw', query_datetime, version_data_table)

        self.set_info('Datetime {} : {} time ranges {} versions ({} wrong) {} points'.format(
            query_datetime,
            len(version_data_table.timestamps),
            len(version_data_table.versions),
            len(wrong_versions),
            len(version_data_table.points)))

        with self.step('Send raw data to Solomon'):
            raw_metrics = processor.VersionDataManager.create_solomon_metrics(
                version_data_table, {})
            self.solomon_send(solomon_project, solomon_cluster,
                              solomon_service_raw, raw_metrics)

        with self.step('Process data'):
            # Get version weight table.
            version_weights = processor.VersionDataManager.get_version_weights(
                processor.MonitorValues.load_history(self.Parameters.config_name, timestamp),
                monitor_config.weight_sensor_name)
            version_weights_table = processor.VersionDataManager.create_version_weights_table(
                timestamp, version_weights)
            self.log_version_data_table('weight', query_datetime, version_weights_table)

            # Get version labels.
            processed_version_labels = processor.WeightAnalyser.get_version_labels(
                monitor_config, version_weights)
            self.log_version_labels_dict('labels', query_datetime, processed_version_labels)

            # Remove ignored versions.
            processed_version_data_table = version_data_table.build_without_versions(
                monitor_config.ignore_versions)

        with self.step('Send processed data to Solomon'):
            processed_metrics = processor.VersionDataManager.create_solomon_metrics(
                processed_version_data_table, processed_version_labels)
            processed_metrics.extend(
                processor.VersionDataManager.create_solomon_metrics(
                    version_weights_table, processed_version_labels))
            self.solomon_send(solomon_project, solomon_cluster,
                              solomon_service_processed, processed_metrics)

        with self.step('Generate data'):
            versions_mapping = processor.WeightAnalyser.get_generated_versions_mapping(
                monitor_config, version_weights)
            self.log_version_mapping('mapping', query_datetime, versions_mapping)
            generated_version_data_table = version_data_table.build_with_version_mapping(
                versions_mapping)
            if monitor_config.sum_version_name:
                logging.info('Generate sum version "%s"', monitor_config.sum_version_name)
                sum_version_data = version_data_table.build_sum(monitor_config.sum_version_name)
                generated_version_data_table = generated_version_data_table.build_merged_with(sum_version_data)
            self.log_version_data_table('generated', query_datetime, generated_version_data_table)

        with self.step('Send generated data to Solomon'):
            generated_metrics = processor.VersionDataManager.create_solomon_metrics(
                generated_version_data_table, {})
            self.solomon_send(solomon_project, solomon_cluster,
                              solomon_service_generated, generated_metrics)
