import os

import yaml


class MonitorConfig(object):
    _KEY_YQL_TEMPLATES = 'yql-templates'
    _KEY_SOLOMON = 'solomon'
    _KEY_PROJECT = 'project'
    _KEY_CLUSTER = 'cluster'
    _KEY_SERVICES = 'services'
    _KEY_RAW = 'raw'
    _KEY_PROCESSED = 'processed'
    _KEY_GENERATED = 'generated'
    _KEY_VERSION_PATTERN = 'version-pattern'
    _KEY_FRESH = 'fresh'
    _KEY_STALE = 'stale'
    _KEY_WEIGHT_SENSOR_NAME = 'weight-sensor-name'
    _KEY_SIGNIFICANT_DAILY_WEIGHT_THRESHOLD = 'significant-daily-weight-threshold'
    _KEY_RELEASE_DAILY_WEIGHT_THRESHOLD = 'release-daily-weight-threshold'
    _KEY_LEADERS_DAILY_WEIGHT_PERCENT = 'leaders-daily-weight-percent'
    _KEY_TOP_VERSIONS_COUNT = 'top-versions-count'
    _KEY_SUM_VERSION_NAME = 'sum-version-name'
    _KEY_IGNORE_VERSIONS = 'ignore-versions'
    _KEY_METRICS = 'metrics'
    _KEY_DEFAULT = 'default'
    _KEY_VALUES = 'values'

    def __init__(self, config_json, yql_templates_dir):
        """
        :type config_json: dict[str, any]
        :type yql_templates_dir: str
        """
        yql_templates = config_json[self._KEY_YQL_TEMPLATES]
        self.yql_fresh_template_path = os.path.join(
            yql_templates_dir, yql_templates[self._KEY_FRESH])
        self.yql_stale_template_path = os.path.join(
            yql_templates_dir, yql_templates[self._KEY_STALE])

        solomon_config = config_json[self._KEY_SOLOMON]
        self.solomon_project = solomon_config[self._KEY_PROJECT]
        self.solomon_cluster = solomon_config[self._KEY_CLUSTER]
        self.solomon_service_raw = solomon_config[self._KEY_SERVICES][self._KEY_RAW]
        self.solomon_service_processed = solomon_config[self._KEY_SERVICES][self._KEY_PROCESSED]
        self.solomon_service_generated = solomon_config[self._KEY_SERVICES][self._KEY_GENERATED]

        self.version_pattern = config_json[self._KEY_VERSION_PATTERN]
        self.weight_sensor_name = config_json[self._KEY_WEIGHT_SENSOR_NAME]
        self.significant_daily_weight_threshold = config_json[self._KEY_SIGNIFICANT_DAILY_WEIGHT_THRESHOLD]
        self.release_daily_weight_threshold = config_json[self._KEY_RELEASE_DAILY_WEIGHT_THRESHOLD]
        self.leaders_daily_weight_percent = config_json[self._KEY_LEADERS_DAILY_WEIGHT_PERCENT]
        self.top_versions_count = config_json[self._KEY_TOP_VERSIONS_COUNT]
        self.sum_version_name = config_json.get(self._KEY_SUM_VERSION_NAME)

        self.ignore_versions = config_json[self._KEY_IGNORE_VERSIONS]

        self.metrics_default = {
            metric_name: metric_json[self._KEY_DEFAULT]
            for metric_name, metric_json in config_json[self._KEY_METRICS].iteritems()
            if self._KEY_DEFAULT in metric_json}
        self.metrics_values = {
            metric_name: metric_json[self._KEY_VALUES]
            for metric_name, metric_json in config_json[self._KEY_METRICS].iteritems()}


class MonitorConfigLoader(object):
    _YAML_EXT = '.yaml'
    _CONFIG_IGNORE_LIST = ['.revision']

    _CONFIG_SCHEMA = {
        'type': 'object',
        'properties': {
            'yql-templates': {
                'type': 'object',
                'properties': {
                    'fresh': {'type': 'string'},
                    'stale': {'type': 'string'},
                },
                'required': ['fresh', 'stale'],
            },

            'solomon': {
                'type': 'object',
                'properties': {
                    'project': {'type': 'string'},
                    'cluster': {'type': 'string'},
                    'services': {
                        'type': 'object',
                        'properties': {
                            'raw': {'type': 'string'},
                            'processed': {'type': 'string'},
                            'generated': {'type': 'string'},
                        },
                        'required': ['raw', 'processed', 'generated'],
                    },
                },
                'required': ['project', 'cluster', 'services'],
            },

            'version-pattern': {'type': 'string'},
            'weight-sensor-name': {'type': 'string'},
            'significant-daily-weight-threshold': {'type': 'integer'},
            'release-daily-weight-threshold': {'type': 'integer'},
            'leaders-daily-weight-percent': {'type': 'integer'},
            'top-versions-count': {'type': 'integer'},
            'sum-version-name': {'type': 'string'},

            'metrics': {
                'type': 'object',
                'patternProperties': {
                    '^.*$': {'$ref': '#/definitions/metric-versions-values'},
                },
            },
            'ignore-versions': {
                'type': 'array',
                'items': {'type': 'string'},
            },
        },

        'required': [
            'yql-templates',

            'solomon',

            'version-pattern',
            'weight-sensor-name',
            'significant-daily-weight-threshold',
            'release-daily-weight-threshold',
            'leaders-daily-weight-percent',
            'top-versions-count',

            'metrics',
            'ignore-versions',
        ],

        'definitions': {
            'metric-versions-values': {
                'type': 'object',
                'properties': {
                    'default': {'type': 'string'},
                    'values': {
                        'type': 'object',
                        'patternProperties': {
                            '^.*$': {
                                'type': 'array',
                                'items': {'type': 'string'},
                            },
                        },
                    },
                },
                'required': [
                    'values',
                ],
            },
        },
    }

    @classmethod
    def available_configs(cls, config_dir=None):
        """
        :type config_dir: str | None
        :return: pairs of strings: config name and config path.
        :rtype: dict[str, str]
        """
        if not config_dir:
            config_dir = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'config')
        try:
            config_names = [n for n in os.listdir(config_dir) if n not in cls._CONFIG_IGNORE_LIST]
        except OSError:
            return {}

        configs = {}
        for config_name in config_names:
            if config_name.endswith(cls._YAML_EXT):
                choice_name = config_name[:-len(cls._YAML_EXT)]
            else:
                raise ValueError('Wrong config extension: {}'.format(config_name))
            configs[choice_name] = os.path.join(config_dir, config_name)

        return configs

    @classmethod
    def _check_template_paths(cls, template_path_list):
        for template_path in template_path_list:
            if not os.path.isfile(template_path):
                raise ValueError('Template does not exist: {}'.format(template_path))

    @classmethod
    def load_config(cls, config_path, yql_templates_dir=None):
        """
        :type config_path: str
        :type yql_templates_dir: str | None
        :rtype: MonitorConfig
        """
        with open(config_path, 'r') as f:
            config_json = yaml.safe_load(f.read())

        # Validate config JSON.
        import jsonschema
        jsonschema.validate(config_json, cls._CONFIG_SCHEMA)

        # Create config object.
        if not yql_templates_dir:
            yql_templates_dir = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'yql-templates')
        monitor_config = MonitorConfig(config_json, yql_templates_dir)

        # Check templates paths.
        cls._check_template_paths([
            monitor_config.yql_fresh_template_path,
            monitor_config.yql_stale_template_path,
        ])

        return monitor_config
