import os

from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.common.types import notification as ctn
from sandbox.sandboxsdk import process

from sandbox.projects.browser.common.git import repositories
from sandbox.projects.browser.common.hpe import HermeticPythonEnvironment


class BrowserPerfFailedBenchmarks(sdk2.Task):
    """Create and actualize tickets for flaked benchmarks"""

    class Parameters(sdk2.Parameters):
        kill_timeout = 10 * 60  # 10min
        _container = sdk2.parameters.Container(
            'Environment container resource',
            default_value=1094621267,
            required=True
        )

        actualize = sdk2.parameters.Bool('Actualize tickets', default=True)

        create = sdk2.parameters.Bool('Create tickets', default=True)
        flakiness = sdk2.parameters.Integer(
            'Flakiness percent', required=True, default=50)
        assignee = sdk2.parameters.String('Assignee')

        days = sdk2.parameters.Integer(
            'Quantity days for flakiness calculation',
            required=True, default=7
        )
        dry_run = sdk2.parameters.Bool('Dry run', default=False)

        token_vault = sdk2.parameters.String(
            'Token vault', required=True, default='speedy-token')
        branch = sdk2.parameters.String(
            'Yin repo branch', required=True, default='master')
        robot_speedy_yav_token = sdk2.parameters.String(
            'YAV TOKEN', required=True, default='ROBOT_SPEEDY_YAV_TOKEN')

        notifications = [
            sdk2.Notification(
                [ctt.Status.FAILURE, ctt.Status.EXCEPTION, ctt.Status.TIMEOUT],
                ['malets', 'victorsamun'],
                ctn.Transport.EMAIL
            )
        ]

    def yin_path(self, *args):
        return str(self.path('yin', *args))

    def on_execute(self):
        repositories.Stardust.yin().clone(
            self.yin_path(), self.Parameters.branch)
        with HermeticPythonEnvironment(
            python_version='3.9.7',
            pip_version='20.3.4',
            requirements_files=[sdk2.Path(self.yin_path('requirements.txt'))]
        ) as hpe:
            os.environ['REQUESTS_CA_BUNDLE'] = '/etc/ssl/certs/'
            os.environ['STARTREK_TOKEN'] = sdk2.Vault.data(
                self.Parameters.token_vault)
            os.environ['ROBOT_SPEEDY_YAV_TOKEN'] = sdk2.Vault.data(
                self.Parameters.robot_speedy_yav_token)
            env = os.environ.copy()
            if 'PYTHONPATH' in env:
                del env['PYTHONPATH']

            cmdline = [
                str(hpe.python_executable),
                '-m', 'yin.perf.failed_benchmarks',
                '-d', str(self.Parameters.days)
            ]

            if self.Parameters.dry_run:
                cmdline.append('--dry-run')

            if not self.Parameters.actualize:
                cmdline.append('--no-actualize')

            if self.Parameters.create:
                cmdline.extend(['-f', str(self.Parameters.flakiness)])
                if self.Parameters.assignee:
                    cmdline.extend(['-a', self.Parameters.assignee])
            else:
                cmdline.append('--no-create')

            process.run_process(
                cmdline,
                log_prefix='create_actualize_tickets',
                outputs_to_one_file=False,
                work_dir=self.yin_path(),
                environment=env
            )
