
import json
import os

from sandbox.common.types import client as ctc
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk import process
from sandbox import sdk2

from sandbox.projects.browser.common.git import repositories
from sandbox.projects.browser.common.hpe import HermeticPythonEnvironment

SCRIPT_NAME = 'histograms.difference'


class DiffJSONResource(sdk2.Resource):
    """fetched json file"""


class BrowserPerfHistogramsDifference(sdk2.Task):
    """
    Calculates difference between histograms for specified
    experiment id and group ids.
    """

    RESULT_JSON = 'result.json'

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC
        environments = [PipEnvironment('virtualenv', '15.1.0')]

    class Parameters(sdk2.Parameters):
        kill_timeout = 24 * 60 * 60  # 24 hours

        branch = sdk2.parameters.String(
            "Performance repo branch", required=True, default='master'
        )
        filters = sdk2.parameters.JSON(
            "Filters", required=True, default='{}'
        )
        experiment = sdk2.parameters.String(
            "Experiment ID", required=True, default='0'
        )
        group1 = sdk2.parameters.String(
            "First group ID", required=True, default='0'
        )
        group2 = sdk2.parameters.String(
            "Second group ID", required=True, default='0'
        )
        instance_type = sdk2.parameters.String(
            "Instance type",
            choices=[(_, _) for _ in ('testing', 'production', 'development')],
            default='testing', required=True,
        )
        publish = sdk2.parameters.Bool(
            "Publish", default=False
        )
        report_url = sdk2.parameters.String(
            "Report URL", default=None
        )

    def my_path(self, *args):
        return str(self.path('performance', *args))

    def on_execute(self):
        os.environ["INSTANCE_TYPE"] = self.Parameters.instance_type
        repositories.Stardust.performance().clone(
            self.my_path(), self.Parameters.branch)
        os.environ['PYTHONPATH'] = \
            '{}:{}'.format(os.environ['PYTHONPATH'], self.my_path())

        with HermeticPythonEnvironment(
            python_version='2.7.17',
            pip_version='9.0.2',
            requirements_files=[sdk2.Path(self.my_path('sandbox', 'requirements.txt'))]
        ) as hpe:

            cmd = [
                [str(hpe.python_executable),
                 self.my_path('sandbox/performanceSandboxScripts/launcher.py')],
                '--script', SCRIPT_NAME,
                '--filters', "'{}'".format(json.dumps(self.Parameters.filters)),
                '--experiment', self.Parameters.experiment,
                '--first_group', self.Parameters.group1,
                '--second_group', self.Parameters.group2,
                '--out_path', self.RESULT_JSON,
            ]

            if self.Parameters.report_url:
                cmd.append(
                    '--report_url {}'.format(self.Parameters.report_url))

            if self.Parameters.publish:
                cmd.append('--publish')

            proc = process.run_process(
                cmd,
                work_dir=self.my_path(),
                log_prefix=SCRIPT_NAME,
                outputs_to_one_file=False,
                shell=True,
            )

            if proc.returncode == 0:
                sdk2.ResourceData(
                    DiffJSONResource(self, 'Results file',
                                     self.my_path(self.RESULT_JSON))
                )
