import contextlib
import contextlib2
import json
import os
import shutil
import subprocess
import tempfile

from sandbox.common.types import resource, client
from sandbox.projects.browser.common.git import Git
from sandbox.projects.browser.common.hpe import HermeticPythonEnvironment
from sandbox.projects.common.teamcity import TeamcityArtifactsContext
from sandbox.projects.resource_types import RELEASABLE_DUMMY
from sandbox.sandboxsdk import process
from sandbox import sdk2


@contextlib.contextmanager
def temporary_file(**tempfile_args):
    file_ = tempfile.NamedTemporaryFile(delete=False, **tempfile_args)
    try:
        file_.close()
        yield file_.name
    finally:
        os.remove(file_.name)


@contextlib.contextmanager
def temporary_directory(**mkdtemp_args):
    directory = tempfile.mkdtemp(**mkdtemp_args)
    try:
        yield directory
    finally:
        shutil.rmtree(directory)


def remove_cr(input_, output):
    with open(input_) as input_file, open(output, 'w') as output_file:
        for line in input_file:
            output_file.write('{}\n'.format(line.strip()))


@contextlib.contextmanager
def docker_engine(setup_script_source):
    with temporary_directory(prefix='docker_root_') as docker_directory:
        with temporary_file(prefix='docker_setup_script_',
                            suffix='.sh') as docker_setup_script:
            remove_cr(setup_script_source, docker_setup_script)

            process.run_process(
                ['bash', '-x', docker_setup_script, docker_directory],
                shell=True, outputs_to_one_file=True,
                log_prefix='docker_setup')
        try:
            # Socket path is hard-coded in docker setup script
            yield "unix://{}".format(
                os.path.join(docker_directory, 'docker.sock'))
        finally:
            process.run_process(
                ['service', 'docker', 'stop'],
                shell=True, outputs_to_one_file=True,
                log_prefix='docker_stop')


class BrowserPerfServerBuild(sdk2.Task):

    class Requirements(sdk2.Requirements):
        privileged = True
        execution_space = 10000
        client_tags = client.Tag.IPV6 & client.Tag.LINUX_XENIAL
        max_restarts = 0

    class Parameters(sdk2.Parameters):
        MASTER_BRANCH = 'master'

        kill_timeout = 35 * 60  # 35min

        _container = sdk2.parameters.Container(
            "Environment container resource",
            platform="linux_ubuntu_16.04_xenial",
            description="Ubuntu Xenial with Docker 19.03.1",
            default_value=1094621267,
            required=True
        )

        branch = sdk2.parameters.String(
            "Performance repo branch", required=True,
            default=MASTER_BRANCH
        )
        commit = sdk2.parameters.String('Commit to build on')

        docker_setup_script = \
            sdk2.parameters.Resource("Script for docker setup",
                                     name="docker_setup_script",
                                     attrs={"config_type": "docker_setup"},
                                     resource_type=RELEASABLE_DUMMY,
                                     state=resource.State.READY, required=True,
                                     default_value='1094939756')

        registry_login = \
            sdk2.parameters.String('Yandex login to use with docker login',
                                   default_value='robot-speedy', required=True)
        vault_item_name = sdk2.parameters.String(
            'Vault item with oauth token for '
            'registry.yandex.net (vault item name)',
            default_value='speedy-registry-token',
            required=True)

        repo_url = sdk2.parameters.String(
            'Url to perf application repo', required=True,
            default_value='https://bitbucket.browser.yandex-team.ru'
                          '/scm/stardust/performance.git')

        targets = sdk2.parameters.List(
            'Targets to build and push',
            default=[])

        push = sdk2.parameters.Bool('Push built images to the registry',
                                    default=True)

        no_pull = sdk2.parameters.Bool(
            'Do not pull cache images from the registry')

    def _auth(self):
        return (self.Parameters.registry_login,
                sdk2.Vault.data(self.Parameters.vault_item_name))

    def _perf_path(self, *args):
        return str(self.path('performance', *args))

    def _branch(self):
        branch = self.Parameters.branch
        return (
            branch.split('/', 2)[-1] if branch.startswith('refs/heads/')
            else branch
        )

    def on_prepare(self):
        Git(self.Parameters.repo_url, filter_branches=False).clone(
            self._perf_path(), self._branch(), commit=self.Parameters.commit)

    def on_execute(self):
        setup_script_path = str(sdk2.ResourceData(
            self.Parameters.docker_setup_script).path)
        with contextlib2.ExitStack() as stack:
            host = stack.enter_context(docker_engine(setup_script_path))

            login, token = self._auth()

            with HermeticPythonEnvironment(
                python_version='3.8.10',
                pip_version='21.1.1',
                requirements_files=[sdk2.Path(
                    self._perf_path('tools/build/requirements.txt')
                )]
            ) as hpe:
                cmdline = [
                    str(hpe.python_executable),
                    self._perf_path('tools/build/build.py'),
                    '--host', host, '--login', login,
                    '--branch', self._branch(),
                    '--commit', self.Parameters.commit
                ]
                if self.Parameters.targets:
                    cmdline.append('--targets')
                    cmdline += self.Parameters.targets

                if self.Parameters.push:
                    cmdline.append('--push')

                if self.Parameters.no_pull:
                    cmdline.append('--no-pull')

                digests_file = stack.enter_context(
                    temporary_file(prefix='push_digests_'))

                env = os.environ.copy()
                env['DOCKER_OAUTH_TOKEN'] = token

                cmdline += ['--teamcity', '--push-digests', digests_file]

                tac = TeamcityArtifactsContext(log_name='docker_build')

                subprocess.check_call(cmdline, env=env, stdout=tac.output,
                                      stderr=subprocess.STDOUT)

                digests = stack.enter_context(open(digests_file))
                for tag, digest, size in json.load(digests):
                    self.set_info('{} {}'.format(tag, digest))
