import os
import logging

from sandbox.common.types.client import Tag
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk import process
from sandbox import sdk2

from sandbox.projects.browser.common.git import repositories
from sandbox.projects.browser.common.hpe import HermeticPythonEnvironment


class BrowserPulseInfo(sdk2.Task):
    """Collect experiments and histograms info for Pulse"""

    class Requirements(sdk2.Requirements):
        environments = [PipEnvironment('virtualenv', '15.1.0')]
        cores = 1  # exactly 1 core
        Tag.GENERIC | Tag.MULTISLOT

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):
        kill_timeout = 10 * 60  # 10 minutes

        _container = sdk2.parameters.Container(
            "Environment container resource",
            default_value=1044237909,
            required=True,
        )

        with sdk2.parameters.Group('General settings') as general_settings:
            pool = sdk2.parameters.String(
                'YT pool',
                choices=[(_, _) for _ in ('robot-pulse', 'browser')],
                default='robot-pulse',
                required=True,
            )
            yt_token = sdk2.parameters.String(
                "YT token vault",
                required=True,
                default='pulse-yt-token',
            )

            tc_token = sdk2.parameters.String(
                "TC token vault",
                required=True,
                default='robot-pulse_browser_teamcity_token',
            )

            branch = sdk2.parameters.String(
                "Bryt repo branch",
                required=True,
                default='master',
            )
            prod = sdk2.parameters.Bool('Prod', default=False)

    def my_path(self, *args):
        return str(self.path('bryt', *args))

    @property
    def logger(self):
        return logging.getLogger(self.__class__.__name__)

    def on_execute(self):
        repositories.Stardust.bryt().clone(
            self.my_path(), self.Parameters.branch,
        )
        with HermeticPythonEnvironment(
            python_version='2.7.17',
            pip_version='9.0.2',
            requirements_files=[sdk2.Path(self.my_path('requirements.txt'))],
            packages=['uatraits']
        ) as hpe:
            hpe.use_cache = False

            os.environ['YT_TOKEN'] = sdk2.Vault.data(self.Parameters.yt_token)
            os.environ['TC_TOKEN'] = sdk2.Vault.data(self.Parameters.tc_token)
            os.environ['YT_DEFAULT_POOL'] = self.Parameters.pool
            os.environ['PYTHONPATH'] = \
                '{}:{}'.format(os.environ['PYTHONPATH'], self.my_path())

            if self.Parameters.prod:
                os.environ['YT_USE_PROD'] = '1'

            process.run_process(
                [str(hpe.python_executable), self.my_path('bryt/bin/pulse_info.py')],
                work_dir=self.my_path(),
                log_prefix='pulse_info',
                outputs_to_one_file=False,
                shell=True,
            )
