import contextlib
import os
import tempfile
from urlparse import urlparse

from sandbox.common.types import client as ctc
import sandbox.common.types.misc as ctm
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk import process
from sandbox import sdk2

from sandbox.projects.browser.common.git import repositories
from sandbox.projects.browser.common.hpe import HermeticPythonEnvironment

WPR_RESOURCE_DESC = 'Automatically recorded wpr_archive.'
CONTAINER_RESOURCE_ID = 611829642
BENCHMARK_OUTPUT_DIR_DESC = 'Output directory for recording artifacts.'

DEFAULT_BUCKET = 'pulse'

TESTING_ACCESS_KEY = 'bsqh9V3JjKe64mQWHAjP'
TESTING_VAULT_NAME = 'robot_speedy_mdst_secret_key'
TESTING_S3_URL = 'http://s3.mdst.yandex.net'

PROD_ACCESS_KEY = 'HghBKLNMUDU9Xd81awyw'
PROD_VAULT_NAME = 'robot_speedy_mds_secret_key'
PROD_S3_URL = 'https://s3.mds.yandex.net'


class WprArchive(sdk2.Resource):
    product = sdk2.Attributes.String('product', required=True)
    snapshot_id = sdk2.Attributes.String('snapshot_id', required=True)
    ttl = 7


class BenchmarkOutputDir(sdk2.Resource):
    ttl = 7


@contextlib.contextmanager
def record_config_from_file(contents):
    with tempfile.NamedTemporaryFile(delete=False) as result:
        result.write(contents)
        result.close()
        try:
            yield result.name
        finally:
            os.unlink(result.name)


class BrowserPerfRecordWprArchive(sdk2.Task):
    """Recording WPR archive for telemetry benchmarks."""

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC
        environments = [PipEnvironment('virtualenv', '15.1.0')]
        ram = 5 * 1024
        disk_space = 20 * 1024
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Parameters):
        kill_timeout = 60 * 60  # 1 hour

        with sdk2.parameters.Group('General settings') as general_settings:
            product = sdk2.parameters.String('Product', required=True)
            snapshot_id = sdk2.parameters.String('SnapshotId',
                                                 required=True)
            record_config = sdk2.parameters.String(
                'Record config', multiline=True, required=True)

            mds_upload = sdk2.parameters.Bool(
                'Upload wpr_archive to S3MDS.  Key format is '
                '{snapshot_id}/wpr_archive.tar',
                default=False)

            record_video = sdk2.parameters.Bool(
                'Record video of benchmark run (if platform allows)',
                default=False)

            s3_bucket = sdk2.parameters.String('S3-MDS bucket', required=True,
                                               default_value=DEFAULT_BUCKET)
            s3_url = sdk2.parameters.String(
                'S3-MDS url', default_value=TESTING_S3_URL,
                choices=[
                    ('testing', TESTING_S3_URL),
                    ('prod', PROD_S3_URL)
                ])
            s3_access_key_id = sdk2.parameters.String('S3-MDS access key id')
            vault_owner = sdk2.parameters.String(
                'S3-MDS secret key vault owner')
            vault_name = sdk2.parameters.String('S3-MDS secret key vault name')

            proxy_url = sdk2.parameters.String('WPR http-proxy url')
            proxy_auth_vault = sdk2.parameters.String(
                'WPR proxy vault with credentials')

            record_product = sdk2.parameters.String(
                'Browser that is used for recording wpr', required=True,
                default='chromium-snapshot')
            spec = sdk2.parameters.String('Specificator for record_product')

            retry = sdk2.parameters.Integer(
                'Retry recording of failed pages. Skip retries by default',
                default=0)
            max_failures = sdk2.parameters.Integer(
                'The value of --max-failures arg to pass to record_wpr',
                default=None)

            branch = sdk2.parameters.String(
                'Yin branch', default_value='master', required=True)

            commit = sdk2.parameters.String('Yin commit')

        _container = sdk2.parameters.Container(
            'Environment container resource',
            default_value=CONTAINER_RESOURCE_ID, required=True)

    def yin_path(self, *args):
        return str(self.path('yin', *args))

    def prepare_s3_parameters(self, cmd, env):
        key = '{}/wpr_archive.tar'.format(self.Parameters.snapshot_id)

        s3_url = self.Parameters.s3_url
        s3_access_key = self.Parameters.s3_access_key_id
        vault_name = self.Parameters.vault_name
        vault_owner = self.Parameters.vault_owner
        if self.Parameters.s3_bucket == DEFAULT_BUCKET and not s3_access_key:
            vault_owner = 'SPEED_INFRA'
            if s3_url == PROD_S3_URL:
                s3_access_key = PROD_ACCESS_KEY
                vault_name = PROD_VAULT_NAME
            else:
                s3_access_key = TESTING_ACCESS_KEY
                vault_name = TESTING_VAULT_NAME

        cmd.extend([
            '--s3-key', key,
            '--s3-bucket', self.Parameters.s3_bucket,
            '--s3-url', s3_url,
        ])
        env['S3_ACCESS_KEY_ID'] = s3_access_key
        s3_secret_key = sdk2.Vault.data(vault_owner, vault_name)
        env['S3_SECRET_KEY'] = s3_secret_key

    def set_wpr_proxy(self, env):
        if self.Parameters.proxy_url:
            proxy_url = self.Parameters.proxy_url
            if self.Parameters.proxy_auth_vault:
                credentials = sdk2.Vault.data(self.Parameters.proxy_auth_vault)
                parsed_url = urlparse(proxy_url)
                assert parsed_url.scheme and parsed_url.netloc
                proxy_url = '{}://{}@{}'.format(parsed_url.scheme,
                                                credentials, parsed_url.netloc)
            env['WPR_HTTP_PROXY'] = proxy_url

    def on_execute(self):
        repositories.Stardust.yin().clone(
            self.yin_path(), self.Parameters.branch,
            commit=self.Parameters.commit)
        with HermeticPythonEnvironment(
            python_version='3.9.7',
            pip_version='20.3.4',
            requirements_files=[sdk2.Path(self.yin_path('requirements.txt'))]
        ) as hpe, record_config_from_file(self.Parameters.record_config) \
                as record_config:

            benchmark_output_dir = os.path.join(self.yin_path(),
                                                'output')
            cmd = [
                str(hpe.python_executable),
                '-m', 'yin.perf.record_wpr_archive',
                '--record-config', record_config,
                '--product', self.Parameters.record_product,
                '--output-dir', benchmark_output_dir,
                '--display', ''
            ]
            if self.Parameters.spec:
                cmd.extend(['--spec', self.Parameters.spec])

            if self.Parameters.retry:
                cmd.extend(['--retry', str(self.Parameters.retry)])

            if self.Parameters.max_failures is not None:
                cmd.extend(['--max-failures',
                            str(self.Parameters.max_failures)])

            if self.Parameters.record_video:
                cmd.extend(['--record-video'])

            env = os.environ.copy()
            if 'PYTHONPATH' in env:
                del env['PYTHONPATH']
            self.set_wpr_proxy(env)
            env['ROBOT_SPEEDY_YAV_TOKEN'] = sdk2.Vault.data(
                'SPEED_INFRA', 'ROBOT_SPEEDY_YAV_TOKEN')
            if self.Parameters.mds_upload:
                self.prepare_s3_parameters(cmd, env)

            outfile = os.path.join(self.yin_path(), 'wpr_archive.tar')
            process.run_process(
                cmd,
                work_dir=self.yin_path(),
                log_prefix='record_wpr_archive',
                outputs_to_one_file=True,
                environment=env,
                shell=False
            )

            resource = WprArchive(
                self, WPR_RESOURCE_DESC, outfile,
                product=self.Parameters.product,
                snapshot_id=self.Parameters.snapshot_id)
            sdk2.ResourceData(resource).ready()

            if os.path.exists(benchmark_output_dir):
                output_dir_resource = BenchmarkOutputDir(
                    self, BENCHMARK_OUTPUT_DIR_DESC, benchmark_output_dir)
                sdk2.ResourceData(output_dir_resource).ready()
