import os
import datetime as dt

from sandbox.common.types import client as ctc
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk import process
from sandbox import sdk2

from sandbox.projects.browser.common.git import repositories
from sandbox.projects.browser.common.hpe import HermeticPythonEnvironment


class BrowserPerfSherlockCalculation(sdk2.Task):
    """Run sherlock calculations."""

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC
        environments = [PipEnvironment('virtualenv', '15.1.0')]

    class Parameters(sdk2.Parameters):
        kill_timeout = 10 * 60 * 60  # 10Hr

        _container = sdk2.parameters.Container(
            "Environment container resource",
            default_value=402010931,
            required=True
        )

        with sdk2.parameters.Group('General settings') as general_settings:
            date = sdk2.parameters.String("Date (YYYY-MM-DD) or yesterday",
                                          required=True)
            calc_name = sdk2.parameters.String(
                'Calculation',
                choices=[(_, _) for _ in ('update_notifications', )],
                default='update_notifications',
                required=True,
            )

            dst_part = sdk2.parameters.String(
                "Destination prefix", required=True,
                default='//home/browser/stasya/')

        with sdk2.parameters.Group('Extra settings') as extra_settings:

            pool = sdk2.parameters.String(
                'YT pool',
                choices=[(_, _) for _ in ('browser', 'robot-browser-yt')],
                default='browser',
                required=True,
            )

            token = sdk2.parameters.String("Token vault", required=True,
                                           default='browser-yt-token')

            branch = sdk2.parameters.String("Bryt repo branch", required=True,
                                            default='master')
            proxy = sdk2.parameters.String(
                'YT proxy name',
                choices=[(_, _) for _ in ('hahn', )],
                default='hahn',
                required=True,
            )

    def my_path(self, *args):
        return str(self.path('bryt', *args))

    def on_execute(self):
        repositories.Stardust.bryt().clone(
            self.my_path(), self.Parameters.branch)

        with HermeticPythonEnvironment(
            python_version='2.7.17',
            pip_version='9.0.2',
            requirements_files=[sdk2.Path(self.my_path('requirements.txt'))],
            packages=['uatraits']
        ) as hpe:

            os.environ['YT_TOKEN'] = sdk2.Vault.data(self.Parameters.token)
            os.environ['PYTHONPATH'] = \
                '{}:{}'.format(os.environ['PYTHONPATH'], self.my_path())

            for _calc_date in self.Parameters.date.split(','):

                calc_date = _calc_date.strip()
                if calc_date.lower() == 'yesterday':
                    calc_date = (dt.datetime.today() - dt.timedelta(days=1))\
                        .strftime('%Y-%m-%d')

                if not calc_date:
                    continue

                cmd = [str(hpe.python_executable),
                       self.my_path('bryt/sherlock/__init__.py'),
                       self.Parameters.calc_name,
                       '--date={}'.format(calc_date),
                       '--pool', self.Parameters.pool,
                       '--dst-part={}'.format(self.Parameters.dst_part),
                       '--proxy={}'.format(self.Parameters.proxy)]

                process.run_process(
                    cmd,
                    work_dir=self.my_path(),
                    log_prefix=self.Parameters.calc_name,
                    outputs_to_one_file=False,
                    shell=True
                )
