import os
import logging
import datetime

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk import process
from sandbox import sdk2

from sandbox.projects.browser.common.git import repositories
from sandbox.projects.browser.common.hpe import HermeticPythonEnvironment


class BrowserPerfYtUmaPulseReport(sdk2.Task):
    """Run uma pulse report"""

    class Requirements(sdk2.Requirements):
        environments = [PipEnvironment('virtualenv', '15.1.0')]
        cores = 1  # exactly 1 core

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):
        kill_timeout = 4 * 60 * 60  # 4Hr

        _container = sdk2.parameters.Container(
            "Environment container resource",
            default_value=402010931,
            required=True
        )

        with sdk2.parameters.Group('General settings') as general_settings:
            pool = sdk2.parameters.String(
                'YT pool',
                choices=[(_, _) for _ in ('robot-pulse', 'browser')],
                default='robot-pulse',
                required=True,
            )

            date = sdk2.parameters.String(
                "Date (YYYY-MM-DDTHH:MM:SS) or lasthour", required=True)

        with sdk2.parameters.Group('Stages') as extra_settings:
            stage_yt = sdk2.parameters.Bool("YT", default=True)
            stage_upload = sdk2.parameters.Bool("Upload", default=True)
            stage_complete = sdk2.parameters.Bool("Complete", default=True)
            ignore_locks = sdk2.parameters.Bool("Ignore locks", default=False)

        with sdk2.parameters.Group('Extra settings') as extra_settings:
            token = sdk2.parameters.String("Token vault", required=True,
                                           default='browser-yt-token')

            stat_user = sdk2.parameters.String("Statface client user name",
                                               required=True,
                                               default='browser-yt-stat-user')

            stat_passw = sdk2.parameters.String("Statface client password",
                                                required=True,
                                                default='browser-yt-stat-pass')
            pulse_token = sdk2.parameters.String(
                'Pulse token vault', required=True,
                default='ROBOT_SPEEDY_PULSE_TOKEN')

            branch = sdk2.parameters.String("Bryt repo branch", required=True,
                                            default='master')

    def my_path(self, *args):
        return str(self.path('bryt', *args))

    @property
    def logger(self):
        return logging.getLogger(self.__class__.__name__)

    def on_execute(self):
        repositories.Stardust.bryt().clone(
            self.my_path(), self.Parameters.branch)
        with HermeticPythonEnvironment(
            python_version='2.7.17',
            pip_version='9.0.2',
            requirements_files=[sdk2.Path(self.my_path('requirements.txt'))],
            packages=['uatraits']
        ) as hpe:

            os.environ['YT_TOKEN'] = sdk2.Vault.data(self.Parameters.token)
            os.environ['STAT_USER'] = \
                sdk2.Vault.data(self.Parameters.stat_user)
            os.environ['STAT_PASSWORD'] = \
                sdk2.Vault.data(self.Parameters.stat_passw)
            os.environ['PULSE_TOKEN'] = \
                sdk2.Vault.data(self.Parameters.pulse_token)
            os.environ['PYTHONPATH'] = \
                '{}:{}'.format(os.environ['PYTHONPATH'], self.my_path())
            if self.Parameters.ignore_locks:
                os.environ['IGNORE_LOCKS'] = '1'
            for _calc_date in self.Parameters.date.split(','):

                calc_date = _calc_date.strip()
                if calc_date == 'lasthour':
                    calc_date = \
                        (datetime.datetime.now() - datetime.timedelta(
                            hours=4)).strftime('%Y-%m-%dT%H:00:00')

                if not calc_date:
                    continue

                cmd_template = [str(hpe.python_executable),
                                self.my_path('bryt/bin/report.py'),
                                'uma_pulse',
                                '--iso-datetime', calc_date,
                                '--sandbox-env',
                                '--pool', self.Parameters.pool]

                if self.Parameters.stage_yt:
                    cmd = cmd_template + ['yt']
                    process.run_process(
                        cmd,
                        work_dir=self.my_path(),
                        log_prefix='uma_yt_{date}'.format(date=calc_date),
                        outputs_to_one_file=False,
                        shell=True
                    )

                if self.Parameters.stage_upload:
                    cmd = cmd_template + ['upload']
                    process.run_process(
                        cmd,
                        work_dir=self.my_path(),
                        log_prefix='uma_upload_{date}'.format(date=calc_date),
                        outputs_to_one_file=False,
                        shell=True
                    )

                if self.Parameters.stage_complete:
                    cmd = cmd_template + ['complete']
                    process.run_process(
                        cmd,
                        work_dir=self.my_path(),
                        log_prefix='uma_compl_{date}'.format(date=calc_date),
                        outputs_to_one_file=False,
                        shell=True
                    )
