import os
import logging

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk import process
from sandbox import sdk2

from sandbox.projects.browser.common.git import repositories
from sandbox.projects.browser.common.hpe import HermeticPythonEnvironment


class BrowserPerfYtUmaPulseReportScheduler(sdk2.Task):
    """Run uma pulse report scheduler"""

    class Requirements(sdk2.Requirements):
        environments = [PipEnvironment('virtualenv', '15.1.0')]
        cores = 1  # exactly 1 core

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):
        kill_timeout = 10 * 60  # 4Hr

        _container = sdk2.parameters.Container(
            "Environment container resource",
            default_value=402010931,
            required=True
        )

        with sdk2.parameters.Group('General settings') as general_settings:
            start_date = sdk2.parameters.String(
                'Date (YYYY-MM-DDTHH:MM:SS)',
                required=True)
            finish_date = sdk2.parameters.String(
                'Date (YYYY-MM-DDTHH:MM:SS)',
                required=False)

        with sdk2.parameters.Group('Extra settings') as extra_settings:
            pulse_token = sdk2.parameters.String(
                "Pulse token (vault)",
                required=True,
                default='robot-pulse-pulse-token')
            yt_token = sdk2.parameters.String(
                "YT token (vault)",
                required=True,
                default='robot-pulse-yt-token')
            sandbox_token = sdk2.parameters.String(
                "Sandbox token (vault)",
                required=True,
                default='robot-pulse-sandbox-token')

            branch = sdk2.parameters.String(
                "Bryt repo branch",
                required=True,
                default='master')
            prod = sdk2.parameters.Bool('Prod', default=False)

    def my_path(self, *args):
        return str(self.path('bryt', *args))

    @property
    def logger(self):
        return logging.getLogger(self.__class__.__name__)

    def on_execute(self):
        repositories.Stardust.bryt().clone(
            self.my_path(), self.Parameters.branch)

        with HermeticPythonEnvironment(
            python_version='2.7.17',
            pip_version='9.0.2',
            requirements_files=[sdk2.Path(self.my_path('requirements.txt'))]
        ) as hpe:
            os.environ['YT_TOKEN'] = \
                sdk2.Vault.data(self.Parameters.yt_token)
            os.environ['PULSE_TOKEN'] = \
                sdk2.Vault.data(self.Parameters.pulse_token)
            os.environ['SANDBOX_TOKEN'] = \
                sdk2.Vault.data(self.Parameters.sandbox_token)
            os.environ['SCHEDULER_START_DATETIME'] = \
                self.Parameters.start_date
            os.environ['PYTHONPATH'] = \
                '{}:{}'.format(os.environ['PYTHONPATH'], self.my_path())
            if self.Parameters.finish_date is not None:
                os.environ['SCHEDULER_FINISH_DATETIME'] = \
                    self.Parameters.finish_date
            if self.Parameters.prod:
                os.environ['YT_USE_PROD'] = '1'

            process.run_process(
                [str(hpe.python_executable), self.my_path('bryt/bin/pulse_scheduler.py')],
                work_dir=self.my_path(),
                log_prefix='uma_scheduler',
                outputs_to_one_file=False,
                shell=True
            )
