
import os
import signal

from sandbox.common.types import client as ctc
from sandbox.common.types import task as ctt
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk import process
from sandbox import sdk2

from sandbox.projects.browser.common.git import repositories
from sandbox.projects.browser.common.hpe import HermeticPythonEnvironment

SEMAPHORE = 'BROWSER_PERF_YASEN_COMMON_SEMAPHORE'
BEFORE_TIMEOUT_STOP_SECONDS = 600  # seconds

SCRIPTS = ['histograms.difference', 'histograms.slices']


class YasenCommonJSONResource(sdk2.Resource):
    """fetched json file"""


class BrowserPerfYasenCommonReport(sdk2.Task):
    """
    Common report task for yasen
    """

    RESULT_JSON = 'result.json'

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC
        environments = [PipEnvironment('virtualenv', '15.1.0')]
        semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name=SEMAPHORE, weight=1, capacity=5)
            ],
        )

    class Parameters(sdk2.Parameters):
        kill_timeout = 24 * 60 * 60  # 24 hours

        branch = sdk2.parameters.String(
            "Performance repo branch", required=True, default='master'
        )
        instance_type = sdk2.parameters.String(
            "Instance type",
            choices=[(_, _) for _ in ('testing', 'production', 'development')],
            default='testing', required=True,
        )
        publish = sdk2.parameters.Bool(
            "Publish", default=False
        )
        config_url = sdk2.parameters.Url(
            "Config URL", default=None
        )
        report_url = sdk2.parameters.Url(
            "Report URL", default=None
        )
        pulse_token = sdk2.parameters.String(
            'Pulse token vault', required=True,
            default='ROBOT_SPEEDY_PULSE_TOKEN')
        script_name = sdk2.parameters.String(
            "Script name",
            choices=[(_, _) for _ in SCRIPTS],
            default=SCRIPTS[0]
        )

        _container = sdk2.parameters.Container(
            "Environment container resource",
            default_value=811509282,
            required=True
        )

    def my_path(self, *args):
        return str(self.path('performance', *args))

    def on_execute(self):
        repositories.Stardust.performance().clone(
            self.my_path(), self.Parameters.branch)

        with HermeticPythonEnvironment(
            python_version='2.7.17',
            pip_version='9.0.2',
            requirements_files=[sdk2.Path(self.my_path('sandbox', 'requirements.txt'))],
        ) as hpe:

            cmd = [
                [str(hpe.python_executable),
                 self.my_path('sandbox/performanceSandboxScripts/launcher.py')],
                '--script', self.Parameters.script_name,
                '--config_url', self.Parameters.config_url,
                '--out_path', self.RESULT_JSON,
            ]

            if self.Parameters.report_url:
                cmd += ["--report_url", self.Parameters.report_url]

            if self.Parameters.publish:
                cmd.append('--publish')

            env = os.environ.copy()
            env["INSTANCE_TYPE"] = self.Parameters.instance_type
            env['PULSE_TOKEN'] = sdk2.Vault.data(self.Parameters.pulse_token)
            env['REQUESTS_CA_BUNDLE'] = "/etc/ssl/certs/"
            env['PYTHONPATH'] = \
                '{}:{}'.format(env['PYTHONPATH'], self.my_path())
            proc = process.run_process(
                cmd,
                work_dir=self.my_path(),
                log_prefix=self.Parameters.script_name,
                outputs_to_one_file=False,
                environment=env,
                shell=True
            )

            if proc.returncode == 0:
                sdk2.ResourceData(
                    YasenCommonJSONResource(
                        self, 'Results file',
                        self.my_path(self.RESULT_JSON))
                )

    def _kill_child_processes(self):
        for child_process in sdk2.helpers.ProcessRegistry:
            try:
                os.kill(child_process.pid, signal.SIGTERM)
            except OSError:
                pass

    def on_break(self, prev_status, status):
        super(BrowserPerfYasenCommonReport, self).on_break(prev_status, status)
        self._kill_child_processes()
