# -*- encoding: utf-8 -*-

import datetime
import os
import subprocess

import pytz

import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.projects.browser.common.hpe import HermeticPythonEnvironment
from sandbox.projects.browser.plus_point_scrapper import processor
from sandbox.sandboxsdk.environments import PipEnvironment

YAV_SECRET_ID = 'sec-01fcxh30whm24pyp5yjtmsemyq'
YQL_TOKEN_KEY = 'yql-token'
TVM_TEST_APP_ID = 2029863
TVM_PROD_APP_ID = 2029799
TVM_TEST_CLIENT_SECRET_ID = 'sec-01fcz9egat7d94fy1h274p22g2'
TVM_PROD_CLIENT_SECRET_ID = 'sec-01fcsvwr20z8636w7cewevzscn'
CLIENT_SECRET_KEY = 'client_secret'
MEDIA_BILLING_TVM_ID_TEST = 2001265
MEDIA_BILLING_TVM_ID_PROD = 2001267

MSK_TIMEZONE = pytz.timezone('Europe/Moscow')
PARAMETER_DATE_FORMAT = '%Y-%m-%d %H:%M:%S'


def localize_date_as_msk(dt):
    return pytz.UTC.localize(dt).astimezone(MSK_TIMEZONE)


def now_msk():
    return localize_date_as_msk(datetime.datetime.utcnow())


def parse_date_msk(date_msk_str):
    return MSK_TIMEZONE.localize(
        datetime.datetime.strptime(date_msk_str, PARAMETER_DATE_FORMAT))


def format_date_msk(date_msk):
    return date_msk.strftime(PARAMETER_DATE_FORMAT)


class TaskProcessor(processor.PassportUserIDProcessor):
    def _handle(self, passport_user_id, utc_start_time, plus_transaction_id):
        start_datetime_utc = datetime.datetime.utcfromtimestamp(utc_start_time)
        start_datetime_msk = localize_date_as_msk(start_datetime_utc)
        self.sandbox_task.set_info(
            'Passport user ID = {}, start time = {} plus transaction id = {}'.format(
                passport_user_id, format_date_msk(start_datetime_msk),
                plus_transaction_id))

        super(TaskProcessor, self)._handle(
            passport_user_id, utc_start_time, plus_transaction_id)


class BrowserPlusPointScrapper(sdk2.Task):
    """
    Searches for users who are deserving of plus points in logs.
    See: https://wiki.yandex-team.ru/browser/dev/infra/docs/pluspointscrapper/
    """

    class Requirements(sdk2.Task.Requirements):
        disk_space = 1024
        cores = 1
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.BROWSER

        environments = (
            PipEnvironment('jsonschema', version='2.6.0'),
            # 'yql' depends on 'yandex-yt' but does not install it.
            PipEnvironment('yandex-yt', '0.9.35'),
            PipEnvironment('yql', '1.2.97'),
        )

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = datetime.timedelta(hours=1).total_seconds()

        dry_run = sdk2.parameters.Bool(
            'Dry run', description='Пробный запуск',
            default=True,
        )

        with sdk2.parameters.Group('YQL') as yql:
            yql_token_secret = sdk2.parameters.YavSecret(
                'YQL token secret', description='YQL-токен',
                default=sdk2.yav.Secret(YAV_SECRET_ID, None, YQL_TOKEN_KEY))

        with sdk2.parameters.Group('TVM') as tvm:
            tvm_test_app_id = sdk2.parameters.Integer(
                'TVM test app ID',
                description='ID тестового TVM-приложения для аутентификации',
                default=TVM_TEST_APP_ID)
            tvm_test_client_secret = sdk2.parameters.YavSecret(
                'TVM test client secret',
                description='Секрет тестового TVM-приложения для аутентификации',
                default=sdk2.yav.Secret(TVM_TEST_CLIENT_SECRET_ID, None, CLIENT_SECRET_KEY))
            tvm_prod_app_id = sdk2.parameters.Integer(
                'TVM prod app ID',
                description='ID боевого TVM-приложения для аутентификации',
                default=TVM_PROD_APP_ID)
            tvm_prod_client_secret = sdk2.parameters.YavSecret(
                'TVM prod client secret',
                description='Секрет боевого TVM-приложения для аутентификации',
                default=sdk2.yav.Secret(TVM_PROD_CLIENT_SECRET_ID, None, CLIENT_SECRET_KEY))

        start_date_msk = sdk2.parameters.String(
            'Start date', description=(
                'Начало периода (время московское) '
                'в формате YYYY-mm-dd HH:MM:SS'),
            default=format_date_msk(now_msk() - datetime.timedelta(hours=6)),
        )
        finish_date_msk = sdk2.parameters.String(
            'Finish date', description=(
                'Окончание периода (время московское) '
                'в формате YYYY-mm-dd HH:MM:SS'),
            default=format_date_msk(now_msk()),
        )

    def get_service_ticket(self, src_tvm_id, src_tvm_secret, dst_tvm_id):
        # TVMAuth library doesn't supported by Sandbox due to
        # error "undefined symbol: PyUnicodeUCS4_FromStringAndSize"
        hpe = HermeticPythonEnvironment(python_version='2.7.17', packages=('tvmauth==3.2.3',))
        with hpe:
            script_path = os.path.join(os.path.dirname(__file__), 'get_tvm_ticket.py')
            return subprocess.check_output([
                str(hpe.python_executable), script_path,
                str(src_tvm_id), src_tvm_secret, str(dst_tvm_id)]).strip()

    def on_execute(self):
        dry_run = self.Parameters.dry_run
        start_date_msk = parse_date_msk(self.Parameters.start_date_msk)
        finish_date_msk = parse_date_msk(self.Parameters.finish_date_msk)
        self.set_info((
            '{}Задан период с {} по {} (время московское).'
        ).format(
            '[Пробный запуск] ' if dry_run else '',
            format_date_msk(start_date_msk), format_date_msk(finish_date_msk),
        ))

        if dry_run:
            src_tvm_id = self.Parameters.tvm_test_app_id
            src_tvm_secret = self.Parameters.tvm_test_client_secret.value()
            dst_tvm_id = MEDIA_BILLING_TVM_ID_TEST
        else:
            src_tvm_id = self.Parameters.tvm_prod_app_id
            src_tvm_secret = self.Parameters.tvm_prod_client_secret.value()
            dst_tvm_id = MEDIA_BILLING_TVM_ID_PROD
        tvm_service_ticket = self.get_service_ticket(src_tvm_id, src_tvm_secret, dst_tvm_id)

        TaskProcessor(
            sandbox_task=self,
            dry_run=dry_run,
            yql_token=self.Parameters.yql_token_secret.value(),
            tvm_service_ticket=tvm_service_ticket,
        ).process(
            start_date_msk=start_date_msk,
            finish_date_msk=finish_date_msk,
        )
