import hashlib
import logging
import os
import uuid

import jinja2
import requests

from sandbox.common import errors as sc_errors
from sandbox.projects.browser.plus_point_scrapper import media_billing_client as mbc

MB_POINTS_CAMPAIGN_ID = 'test_campaign'
MB_POINTS_AMOUNT = 300
MB_PAYLOAD = {
    'db': 'points',
    'issuer': 'testing',
    'cashback_service': 'plus',
    'campaign_name': 'test',
    'cashback_type': 'nontransaction',
}


def create_session_with_retry():
    session = requests.Session()
    retry = requests.packages.urllib3.util.retry.Retry(
        total=5, backoff_factor=0.3,
        method_whitelist=frozenset(['GET', 'POST']))
    adapter = requests.adapters.HTTPAdapter(max_retries=retry)
    session.mount('http://', adapter)
    session.mount('https://', adapter)
    return session


def generate_transaction_id(passport_user_id):
    """
    Generates transaction ID by user ID.
    Transaction ID should be the same each time
    to prevent the second transaction.

    :type passport_user_id: int
    :rtype: str
    """
    assert isinstance(passport_user_id, int)
    sha256_hash_string = 'Transaction for user {}'.format(passport_user_id)
    sha256_hash = hashlib.sha256(sha256_hash_string.encode('ASCII'))
    return str(uuid.UUID(bytes=sha256_hash.digest()[:16]))


class PassportUserIDProcessor(object):
    _MAX_PUID_LIST_SIZE_FOR_SAVE = 1024

    def __init__(self, sandbox_task, dry_run, yql_token, tvm_service_ticket):
        """
        :type sandbox_task: sandbox.sdk2.Task
        :type dry_run: bool
        :type yql_token: str
        :type tvm_service_ticket: str
        """
        self.sandbox_task = sandbox_task
        self.dry_run = dry_run
        self.yql_token = yql_token
        self.tvm_service_ticket = tvm_service_ticket
        self.session = create_session_with_retry()

    def _format_yql_template(self, template_name, **kwargs):
        template_path = os.path.join(
            os.path.dirname(os.path.abspath(__file__)),
            'yql', template_name)
        with open(template_path, 'r') as f:
            env = jinja2.Environment(
                loader=jinja2.BaseLoader,
                trim_blocks=True,
                lstrip_blocks=True)
            template_text = f.read().decode('utf-8')
            sandbox_task_url = 'https://sandbox.yandex-team.ru/task/{}/view'.format(
                self.sandbox_task.id)
            return env.from_string(template_text).render(
                sandbox_task_url=sandbox_task_url,
                **kwargs)

    @classmethod
    def _run(cls, yql_client, yql_query_text):
        yql_query = yql_client.query(query=yql_query_text, syntax_version=1)
        yql_query.run()
        yql_query.get_results(wait=True)
        if not yql_query.is_success:
            raise sc_errors.TaskError('YQL query failed: {}'.format(yql_query.status))
        return yql_query

    def _yql_insert_to_history(self, yql_client, rows):
        """
        :type yql_client: yql.api.v1.client.YqlClient
        :param rows: list of tuple(PassportUserId, UTCStartTime, PlusTransactionId)
        :type rows: list[(int, int, str)]
        """
        return self._run(yql_client, self._format_yql_template(
            'insert-to-history.yql.jinja',
            rows=rows))

    def _yql_select_puids(self, yql_client,
                          start_date_msk, finish_date_msk):
        """
        :type yql_client: yql.api.v1.client.YqlClient
        :type start_date_msk: datetime.datetime
        :type finish_date_msk: datetime.datetime
        """
        return self._run(yql_client, self._format_yql_template(
            'select-puids.yql.jinja',
            start_date_msk=start_date_msk, finish_date_msk=finish_date_msk))

    def _handle(self, passport_user_id, utc_start_time, plus_transaction_id):
        """
        :type passport_user_id: int
        :type utc_start_time: int
        :type plus_transaction_id: str
        """
        logging.info(
            'Schedule transaction %s of user %s on %s server ...',
            plus_transaction_id, passport_user_id,
            'test' if self.dry_run else 'prod')
        mbc.schedule_transaction(
            session=self.session,
            service_ticket=self.tvm_service_ticket,
            production=not self.dry_run,
            user_id=passport_user_id,
            transaction_id=plus_transaction_id,
            points_campaign_id=MB_POINTS_CAMPAIGN_ID,
            points_amount=MB_POINTS_AMOUNT,
            payload=MB_PAYLOAD,
        )

    def _save_to_history(self, yql_client, rows):
        """
        :type yql_client: yql.api.v1.client.YqlClient
        :type rows: list[(int, int, str)]
        """
        if self.dry_run:
            logging.info('[DRY] Save %d PUIDs to history ...', len(rows))
        else:
            logging.info('Save %d PUIDs to history ...', len(rows))
            self._yql_insert_to_history(yql_client, rows)

    def process(self, start_date_msk, finish_date_msk):
        from yql.api.v1.client import YqlClient
        with YqlClient(token=self.yql_token) as yql_client:
            logging.info('Select passport user IDs ...')
            yql_query = self._yql_select_puids(
                yql_client,
                start_date_msk=start_date_msk,
                finish_date_msk=finish_date_msk)

            yql_query.table.fetch_full_data()

            try:
                rows = []
                logging.info('Got %d passport user IDs', len(yql_query.table.rows))
                for passport_user_id, utc_start_time in yql_query.table.rows:
                    plus_transaction_id = generate_transaction_id(int(passport_user_id))
                    logging.info(
                        'Processing PUID: %s (UTC start time = %s, '
                        'generated transaction ID = %s)',
                        passport_user_id, utc_start_time, plus_transaction_id)
                    self._handle(passport_user_id, utc_start_time, plus_transaction_id)
                    rows.append((passport_user_id, utc_start_time, plus_transaction_id))

                    if len(rows) >= self._MAX_PUID_LIST_SIZE_FOR_SAVE:
                        self._save_to_history(yql_client, rows)
                        rows = []
            finally:
                if rows:
                    self._save_to_history(yql_client, rows)
