import logging
import re
from time import time

from sandbox import common
import sandbox.common.types.client as ctc
from sandbox.projects.browser.common import bitbucket as bb
from sandbox.projects.browser.common.git import ConfigureGitEnvironment, GitEnvironment, repositories
from sandbox.projects.common import decorators
from sandbox import sdk2
from sandbox.sdk2.helpers import process


BB_PROJECT = 'stardust'
BB_REPO = 'browser-brandings'
VERSION_PATTERN = re.compile(r'\d+\.\d+\.\d+')


def git_check(*args, **kwargs):
    command_line = ['git'] + list(args)
    with process.ProcessLog(task=sdk2.task.Task.current, logger='git') as pl:
        process.subprocess.check_call(command_line, stdout=pl.stdout, stderr=pl.stderr, **kwargs)


def nothing_to_commit(**kwargs):
    with process.ProcessLog(task=sdk2.task.Task.current, logger='git') as pl:
        return not process.subprocess.check_output(['git', 'status', '--porcelain'], stderr=pl.stderr, **kwargs)


@decorators.retries(3, delay=30, backoff=2, exceptions=(process.subprocess.CalledProcessError,))
def retried_git(*args, **kwargs):
    git_check(*args, **kwargs)


class CreateBrowserBrandingsBranches(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 2 * 1024
        client_tags = ctc.Tag.BROWSER
        environments = (
            GitEnvironment('2.24.1'),
            ConfigureGitEnvironment(email='robot-bro-shuttle@yandex-team.ru', username='robot-bro-shuttle'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 60 * 60
        new_branch_names = sdk2.parameters.List('New branch names')
        previous_branch_name = sdk2.parameters.String('Previous branch name', required=True)
        test_run = sdk2.parameters.Bool('Use test bitbucket')

        with sdk2.parameters.Group('PR settings') as pr_settings:
            commit_msg = sdk2.parameters.String('Commit message', default='Add brands from master branch')
            pr_title = sdk2.parameters.String('Pull request title', default='Add brands from master branch')
            pr_description = sdk2.parameters.String('Pull request description', default='')
            pr_reviewers = sdk2.parameters.List('Reviewers')

        with sdk2.parameters.Group('Credentials') as credentials_group:
            oauth_vault = sdk2.parameters.String(
                'Vault item with oauth token for bitbucket',
                default='robot-bro-shuttle_bitbucket_token')
            ssh_key_vault = sdk2.parameters.String(
                'Vault item with ssh key for bitbucket',
                default='robot-bro-shuttle_ssh_key')

    @common.utils.singleton_property
    def bitbucket(self):
        url = bb.TESTING_BITBUCKET_URL if self.Parameters.test_run else bb.DEFAULT_BITBUCKET_URL
        return bb.BitBucket(url, 'x-oauth-token', sdk2.Vault.data(self.Parameters.oauth_vault))

    @property
    def repo_path(self):
        return str(self.path('brandings'))

    def create_new_brandings_branch(self, from_branch, new_branch, pr_branch):
        git_check('fetch', 'origin', from_branch, cwd=self.repo_path)
        git_check('checkout', from_branch, cwd=self.repo_path)
        git_check('checkout', '-b', new_branch, cwd=self.repo_path)
        retried_git('push', '--set-upstream', 'origin', new_branch, cwd=self.repo_path)
        logging.info('created branch %s', new_branch)

        git_check('checkout', '-b', pr_branch, cwd=self.repo_path)
        git_check('checkout', 'master', '--', 'brands', cwd=self.repo_path)
        if nothing_to_commit(cwd=self.repo_path):
            logging.info('nothing to commit from "brands" folder from master branch, skipping pr')
            return
        git_check('commit', '-m', self.Parameters.commit_msg, cwd=self.repo_path)
        retried_git('push', '--set-upstream', 'origin', pr_branch, cwd=self.repo_path)
        logging.info('added brands to branch %s', pr_branch)

        self.bitbucket.create_pr(
            BB_PROJECT, BB_REPO, self.Parameters.pr_title, self.Parameters.pr_description,
            pr_branch, new_branch, self.Parameters.pr_reviewers)

    def on_execute(self):
        timestamp = int(time())
        repositories.Stardust.browser_brandings(testing=self.Parameters.test_run).clone(
            self.repo_path, branch='master')
        for new_branch_name in self.Parameters.new_branch_names:
            m = VERSION_PATTERN.search(new_branch_name)
            version = m.group() if m else new_branch_name
            pr_branch_name = 'wp-{}/new-branch/{}'.format(version, timestamp)

            with sdk2.ssh.Key(self, self.Parameters.ssh_key_vault, None):
                self.create_new_brandings_branch(self.Parameters.previous_branch_name,
                                                 new_branch_name, pr_branch_name)
