"""
Author: Anatoly Matyukhin <amatyukhin@yandex-team.ru>
"""
import logging
import subprocess

import sandbox.common.types.client as ctc
from sandbox.common import utils

from sandbox.projects.browser.common.git import GitEnvironment, repositories
from sandbox.projects.browser.release_cycle.TransitBrowserIssues import branch_creation as bc
from sandbox.projects.browser.release_cycle.TransitBrowserIssues import cf
from sandbox.projects.browser.release_cycle.TransitBrowserIssues import ff2
from sandbox.projects.browser.release_cycle.TransitBrowserIssues import android_code_freeze
from sandbox.projects.browser.release_cycle.TransitBrowserIssues import android_cut_off

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox import sdk2


class TransitBrowserIssues(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 2 * 1024
        client_tags = ctc.Tag.BROWSER
        environments = (
            PipEnvironment('startrek-client', '1.8.0', custom_parameters=['--upgrade-strategy', 'only-if-needed']),
            GitEnvironment('2.24.1'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 60 * 60

        dry_run = sdk2.parameters.Bool('Dry run')

        with sdk2.parameters.String('Milestone', required=True) as milestone:
            milestone.values.rc_branch_creation = milestone.Value('Release branch creation')
            milestone.values.branch_creation_on_major_rc = milestone.Value('Branch creation in major release')
            milestone.values.ff2 = milestone.Value('Second feature freeze')
            milestone.values.cf = milestone.Value('Code freeze')
            milestone.values.android_cut_off = milestone.Value('Android cut-off')
            milestone.values.android_code_freeze = milestone.Value('Android code freeze')

        with milestone.value.rc_branch_creation:
            rc_version = sdk2.parameters.String('Current release version', required=True)
            rc_branch = sdk2.parameters.String('Current release branch', required=True)
            rc_branch_base = sdk2.parameters.String(
                'Branch from which the current release branch was created', required=True)
            previous_rc_branch = sdk2.parameters.String('Previous release branch', required=True)
            previous_rc_branch_base = sdk2.parameters.String(
                'Branch from which the previous release branch was created', required=True)

        with milestone.value.branch_creation_on_major_rc:
            major_version = sdk2.parameters.String('Current browser major version', required=True)
            previous_versions = sdk2.parameters.List('Browser versions from previous group', required=True)
            chromium_version = sdk2.parameters.Integer('Current chromium version', required=True)

        with milestone.value.ff2, milestone.value.cf, \
                milestone.value.android_cut_off, milestone.value.android_code_freeze:
            version = sdk2.parameters.String('Current browser version', required=True)
            new_version = sdk2.parameters.String('New browser version', required=True)

        sleep_before_start = sdk2.parameters.Integer('Sleep before start (in seconds)')
        startrek_token_vault = sdk2.parameters.String('Vault item with Startrek token',
                                                      default='robot-bro-shuttle_startrek_token')

    def on_enqueue(self):
        sleep_interval = self.Parameters.sleep_before_start
        if sleep_interval > 0:
            self.Parameters.sleep_before_start = 0
            raise sdk2.WaitTime(sleep_interval)

    @utils.singleton_property
    def startrek(self):
        import startrek_client
        return startrek_client.Startrek(
            'TransitBrowserIssues', sdk2.Vault.data(self.Parameters.startrek_token_vault),
            headers={'Accept-Language': 'en-US, en'}, timeout=10)

    def transit_on_rc_branch_creation(self):
        vcs_root = repositories.Stardust.browser()
        vcs_root.update_cache_repo(
            self.Parameters.rc_branch, self.Parameters.rc_branch_base,
            self.Parameters.previous_rc_branch, self.Parameters.previous_rc_branch_base)
        repo_dir = vcs_root.cache_repo_dir

        def merge_base(ref_1, ref_2):
            result = subprocess.check_output(
                ['git', 'merge-base', ref_1, ref_2], cwd=repo_dir).strip()
            logging.info('Merge base of "%s" and "%s" is "%s"', ref_1, ref_2, result)
            return result

        rc_merge_base = merge_base(self.Parameters.rc_branch_base,
                                   self.Parameters.rc_branch)
        prev_rc_merge_base = merge_base(self.Parameters.previous_rc_branch_base,
                                        self.Parameters.previous_rc_branch)

        if merge_base(prev_rc_merge_base, rc_merge_base) != prev_rc_merge_base:
            raise RuntimeError(
                'Commit of previous "{}" branch creation ({}) is not a straight parent '
                'of current "{}" branch creation commit ({})'.format(
                    self.Parameters.previous_rc_branch, prev_rc_merge_base,
                    self.Parameters.rc_branch, rc_merge_base,
                )
            )

        with sdk2.helpers.ProgressMeter('Apply DevRangeRule'):
            bc.DevRangeRule(
                repo_dir, prev_rc_merge_base, rc_merge_base,
                version=self.Parameters.rc_version,
                startrek=self.startrek,
                dry_run=self.Parameters.dry_run,
            ).apply()

    def transit_on_major_branch_creation(self):
        with sdk2.helpers.ProgressMeter('Ensure version in BROWSER queue'):
            bc.ensure_version_in_browser_queue(self.startrek, self.Parameters.major_version, self.Parameters.dry_run)

        with sdk2.helpers.ProgressMeter('Apply MergeChromiumRule'):
            bc.MergeChromiumRule(self.Parameters.chromium_version, self.Parameters.major_version,
                                 self.startrek, self.Parameters.dry_run).apply()

        with sdk2.helpers.ProgressMeter('Apply UnresolvedRule'):
            bc.UnresolvedRule(self.Parameters.previous_versions, self.Parameters.chromium_version,
                              self.Parameters.major_version, self.startrek, self.Parameters.dry_run).apply()

        with sdk2.helpers.ProgressMeter('Apply SanityRule'):
            bc.SanityRule(self.Parameters.chromium_version, self.Parameters.major_version,
                          self.startrek, self.Parameters.dry_run).apply()

        with sdk2.helpers.ProgressMeter('Apply MergeTestsRule'):
            bc.MergeTestsRule(self.Parameters.chromium_version, self.Parameters.major_version,
                              self.startrek, self.Parameters.dry_run).apply()

        with sdk2.helpers.ProgressMeter('Apply AffectedVersionRule'):
            bc.AffectedVersionRule(self.Parameters.chromium_version, self.Parameters.major_version,
                                   self.startrek, self.Parameters.dry_run).apply()

        with sdk2.helpers.ProgressMeter('Apply CrashesRule'):
            bc.CrashesRule(self.Parameters.chromium_version, self.Parameters.major_version,
                           self.startrek, self.Parameters.dry_run).apply()

    def transit_on_ff2(self):
        with sdk2.helpers.ProgressMeter('Apply FF2TransitRule'):
            ff2.FF2TransitRule(self.Parameters.new_version, self.Parameters.version, self.startrek,
                               self.Parameters.dry_run).apply()

    def transit_on_cf(self):
        with sdk2.helpers.ProgressMeter('Apply CFTransitRule'):
            cf.CFTransitRule(self.Parameters.new_version, self.Parameters.version, self.startrek,
                             self.Parameters.dry_run).apply()

    def transit_on_android_cut_off(self):
        with sdk2.helpers.ProgressMeter('Apply android IssuesRule'):
            android_cut_off.IssuesRule(self.Parameters.new_version, self.Parameters.version, self.startrek,
                                       self.Parameters.dry_run).apply()
        with sdk2.helpers.ProgressMeter('Apply android CrashesRule'):
            android_cut_off.CrashesRule(self.Parameters.new_version, self.Parameters.version, self.startrek,
                                        self.Parameters.dry_run).apply()

    def transit_on_android_code_freeze(self):
        with sdk2.helpers.ProgressMeter('Apply android BugsRule'):
            android_code_freeze.BugsRule(self.Parameters.new_version, self.Parameters.version, self.startrek,
                                         self.Parameters.dry_run).apply()

    def on_execute(self):
        {
            'rc_branch_creation': self.transit_on_rc_branch_creation,
            'branch_creation_on_major_rc': self.transit_on_major_branch_creation,
            'ff2': self.transit_on_ff2,
            'cf': self.transit_on_cf,
            'android_cut_off': self.transit_on_android_cut_off,
            'android_code_freeze': self.transit_on_android_code_freeze,
        }[self.Parameters.milestone]()
