# -*- coding: utf-8 -*-
"""
Author: Anatoly Matyukhin <amatyukhin@yandex-team.ru>
"""
import abc
import logging
import re
import subprocess

from sandbox.common import utils

from sandbox.projects.browser.release_cycle.TransitBrowserIssues import common as tbi_common


BROWSER_ISSUES_RE = re.compile('BROWSER-[0-9]+')
PRE_VERSION_RE = re.compile(r'^\d+.\d+/pre$')
IGNORE_COMPONENTS = {
    'Full_Tests',
    'Regression',
}

DEV_MASTER_VERSIONS = ('Dev', 'Master')
MASTER_NEXT_VERSION = 'MasterNext'


@utils.singleton
def _cached_component(startrek, component_id):
    return startrek.components[component_id]


@utils.singleton
def _cached_version(startrek, version_id):
    return startrek.versions[version_id]


@utils.singleton
def _cached_resolution(startrek, resolution_id):
    return startrek.resolutions[resolution_id]


@utils.singleton
def _cached_status(startrek, status_id):
    return startrek.statuses[status_id]


def _get_components(startrek, issue):
    return {_cached_component(startrek, c.id).name for c in issue.components}


def _has_empty_or_dev_or_master_affected_version(startrek, issue):
    versions = [_cached_version(startrek, v.id).name for v in issue.affectedVersions]
    return set(versions) <= {'Dev', 'Master'}


def _get_fix_versions(startrek, issue):
    return [_cached_version(startrek, v.id).name for v in issue.fixVersions]


def _has_empty_or_dev_or_master_fix_version(startrek, issue):
    return set(_get_fix_versions(startrek, issue)) <= set(DEV_MASTER_VERSIONS)


def _has_pre_fix_version(startrek, issue):
    return any(PRE_VERSION_RE.match(version) for version in _get_fix_versions(startrek, issue))


def _is_resolved(startrek, issue):
    status = _cached_status(startrek, issue.status.id).name
    resolution = _cached_resolution(startrek, issue.resolution.id).name if issue.resolution else None
    return (status in ('Resolved', 'Testing', 'Tested', 'Resolved Blocked') or
            (status == 'Closed' and resolution == 'Fixed'))


_RESOLVED_QUERY = ('((Status: "Resolved", "Testing", "Tested", "Resolved Blocked") or '
                   '(Status: "Closed" and Resolution: "Fixed"))')


def _ignored_merge_tag(chromium_version):
    return 'merge{}'.format(chromium_version + 1)


def _issues_from_commit_range(startrek, repo_dir, begin_commit, end_commit):
    from startrek_client.exceptions import Forbidden, NotFound

    commit_messages = subprocess.check_output(
        ['git', 'log', '{}..{}'.format(begin_commit, end_commit), '--first-parent', '--format=\t%B'],
        cwd=repo_dir).split('\t')
    issue_keys = set()
    for message in commit_messages:
        issue_keys.update(BROWSER_ISSUES_RE.findall(message))
    for issue_key in issue_keys:
        try:
            yield startrek.issues[issue_key]
        except (Forbidden, NotFound):
            logging.exception('Failed to get %s:', issue_key)


def ensure_version_in_browser_queue(startrek, version, dry_run=False):
    """
    Creates specified `version` in BROWSER queue if it does not exist.

    :type startrek: startrek_client.client.Startrek
    :type version: str
    :type dry_run: bool
    """
    browser_queue = startrek.queues['BROWSER']
    st_versions = [st_version.name for st_version in browser_queue.versions]

    if version in st_versions:
        logging.info('Version %s already exists.', version)
        return

    logging.info('Creating %s version...', version)
    if dry_run:
        logging.info('Skip creating version because of dry run mode.')
        return
    startrek.versions.create(queue=browser_queue, name=version, description='master-{}/rc'.format(version))


class CommitsRangeRule(tbi_common.TransitRule):
    """
    Process issues mentioned in description of commits from specified commits range.
    """
    __metaclass__ = abc.ABCMeta

    def __init__(self, repo_dir, begin_commit, end_commit, *args, **kwargs):
        """
        :type repo_dir: str
        :type begin_commit: str
        :type end_commit: str
        """
        self._repo_dir = repo_dir
        self._begin_commit = begin_commit
        self._end_commit = end_commit
        super(CommitsRangeRule, self).__init__(*args, **kwargs)

    def _is_target_issue(self, issue):
        return not _has_pre_fix_version(self._startrek, issue)

    def _list_issues(self):
        issues = _issues_from_commit_range(
            self._startrek, self._repo_dir, self._begin_commit, self._end_commit)
        return filter(self._is_target_issue, issues)


class DevRangeRule(CommitsRangeRule):
    """
    Find issues from commits range.
    Set actual fix version to those issues that have only Dev or Master or empty fix version.
    """
    TAG_FORMAT = 'nb_pre_{version}'

    def _is_target_issue(self, issue):
        return (super(DevRangeRule, self)._is_target_issue(issue) and
                _has_empty_or_dev_or_master_fix_version(self._startrek, issue))

    def _update_rules(self):
        return dict(
            fixVersions={'add': self._version, 'remove': DEV_MASTER_VERSIONS},
            tags={'add': self._tag()},
        )

    def _tag(self):
        return self.TAG_FORMAT.format(version=self._version)


class QueryBasedRule(tbi_common.QueryBasedRule):
    __metaclass__ = abc.ABCMeta

    def _filter_issues(self, issue):
        return (
            super(QueryBasedRule, self)._filter_issues(issue) and
            not _has_pre_fix_version(self._startrek, issue)
        )


class IgnoredMergeRule(QueryBasedRule):
    __metaclass__ = abc.ABCMeta

    def __init__(self, chromium_version, *args, **kwargs):
        self._ignored_tag = _ignored_merge_tag(chromium_version)
        super(IgnoredMergeRule, self).__init__(*args, **kwargs)

    def _query_components(self):
        return super(IgnoredMergeRule, self)._query_components() + [
            'Tags: !"{}"'.format(self._ignored_tag)
        ]


class SanityRule(IgnoredMergeRule):
    IGNORE_COMPONENTS = IGNORE_COMPONENTS

    def _query_components(self):
        return super(SanityRule, self)._query_components() + [
            '"Fix Version": empty(), "Dev", "Master"',
            'Status: !"Closed"',
            'Status: !"Tested"',
            'Type: Bug, Sub-bug',
            'Priority: Critical, Blocker',
            'Components: "Sanity"',
        ]

    def _filter_issues(self, issue):
        return (
            super(SanityRule, self)._filter_issues(issue) and
            _has_empty_or_dev_or_master_fix_version(self._startrek, issue)
        )

    def _update_rules(self):
        return dict(
            fixVersions={'add': self._version, 'remove': ['Dev', 'Master']},
            tags={'add': 'nb_sanity_{}'.format(self._version)},
        )


class MergeTestsRule(IgnoredMergeRule):
    IGNORE_COMPONENTS = IGNORE_COMPONENTS

    def _query_components(self):
        return super(MergeTestsRule, self)._query_components() + [
            '"Fix Version": empty(), "Dev", "Master"',
            'Status: !"Closed"',
            'Status: !"Tested"',
            'Type: Bug, Sub-bug',
            'Priority: Critical, Blocker',
        ]

    def _filter_issues(self, issue):
        return (
            super(MergeTestsRule, self)._filter_issues(issue) and
            _has_empty_or_dev_or_master_fix_version(self._startrek, issue) and
            {'Merge', 'Unit-tests'} <= _get_components(self._startrek, issue)
        )

    def _update_rules(self):
        return dict(
            fixVersions={'add': self._version, 'remove': ['Dev', 'Master']},
            tags={'add': 'nb_mergetests_{}'.format(self._version)},
        )


class MergeChromiumRule(QueryBasedRule):
    IGNORE_COMPONENTS = IGNORE_COMPONENTS

    def __init__(self, chromium_version, *args, **kwargs):
        self._chromium_version = chromium_version
        super(MergeChromiumRule, self).__init__(*args, **kwargs)

    def _query_components(self):
        return super(MergeChromiumRule, self)._query_components() + [
            '"Fix Version": !"{}"'.format(self._version),
            'Tags: merge{}'.format(self._chromium_version),
        ]

    def _update_rules(self):
        return dict(
            fixVersions={'add': self._version, 'remove': DEV_MASTER_VERSIONS + (MASTER_NEXT_VERSION,)},
            tags={'add': 'nb_merge_{}'.format(self._version)},
        )


class CrashesRule(IgnoredMergeRule):
    IGNORE_COMPONENTS = IGNORE_COMPONENTS

    def _query_components(self):
        return super(CrashesRule, self)._query_components() + [
            '"Fix Version": empty(), "Dev", "Master"',
            'Status: !"Closed"',
            'Status: !"Tested"',
            'Type: Crash',
            'Priority: Critical',
        ]

    def _filter_issues(self, issue):
        return (
            super(CrashesRule, self)._filter_issues(issue) and
            _has_empty_or_dev_or_master_fix_version(self._startrek, issue)
        )

    def _update_rules(self):
        return dict(
            fixVersions={'add': self._version, 'remove': ['Dev', 'Master']},
            tags={'add': 'nb_crashes_{}'.format(self._version)},
        )


class UnresolvedRule(IgnoredMergeRule):
    IGNORE_COMPONENTS = IGNORE_COMPONENTS

    def __init__(self, previous_versions, *args, **kwargs):
        """
        :type previous_versions: list[str]
        """
        self._previous_versions = previous_versions
        super(UnresolvedRule, self).__init__(*args, **kwargs)

    def _query_components(self):
        return super(UnresolvedRule, self)._query_components() + [
            '"Fix Version": {}'.format(', '.join('"{}"'.format(v) for v in self._previous_versions)),
            'Status: !"Closed"',
            'Status: !"Tested"',
        ]

    def _update_rules(self):
        return dict(
            fixVersions={'add': self._version},
            tags={'add': 'nb_unresolved_{}'.format(self._version)},
        )


class AffectedVersionRule(IgnoredMergeRule):
    IGNORE_COMPONENTS = IGNORE_COMPONENTS

    def _query_components(self):
        return super(AffectedVersionRule, self)._query_components() + [
            '"Affected Version": empty(), "Dev", "Master"',
            'Status: !"Closed"',
            'Status: !"Tested"',
            'Type: Bug, Sub-bug',
        ]

    def _filter_issues(self, issue):
        return (
            super(AffectedVersionRule, self)._filter_issues(issue) and
            _has_empty_or_dev_or_master_affected_version(self._startrek, issue)
        )

    def _update_rules(self):
        return dict(
            affectedVersions={'add': self._version, 'remove': ['Dev', 'Master']},
            tags={'add': 'nb_av_{}'.format(self._version)},
        )
