"""
Author: Anatoly Matyukhin <amatyukhin@yandex-team.ru>
"""
import abc
import json
import logging

from sandbox.projects.common import decorators


def query_startrek(startrek, query):
    from startrek_client.collections import Collection
    from startrek_client.exceptions import StartrekRequestError
    from startrek_client.objects import PaginatedList

    @decorators.retries(3, exceptions=StartrekRequestError)
    def startrek_req():
        logging.info('Query Startrek: "%s"', query)
        result = startrek.issues.find(query=query, perScroll=250, scrollType='sorted')
        if isinstance(result, (Collection, PaginatedList)):
            result = list(result)
        return result

    return startrek_req()


class TransitRule(object):
    __metaclass__ = abc.ABCMeta

    def __init__(self, version, startrek, dry_run=False):
        """
        :type version: str
        :type startrek: startrek_client.client.Startrek
        :type dry_run: bool
        """
        self._version = version
        self._startrek = startrek
        self._dry_run = dry_run

        self._issues = self._list_issues()
        logging.info('%s should be applied to: %s', self.__class__.__name__,
                     ', '.join(issue.key for issue in self._issues))

    @abc.abstractmethod
    def _list_issues(self):
        raise NotImplementedError()

    @abc.abstractmethod
    def _update_rules(self):
        raise NotImplementedError()

    def apply(self):
        rules = self._update_rules()
        logging.info('Apply %s:\n%s', self.__class__.__name__, json.dumps(rules, indent=2))

        if self._dry_run:
            logging.info('Skip applying because of dry run mode.')
            return
        if not self._issues:
            return

        from startrek_client.exceptions import NotFound
        @decorators.retries(5, delay=0.1, backoff=1.5, exceptions=NotFound)
        def wait_for_bulkchange(bulkchange):
            _bulkchange = bulkchange.wait()
            if _bulkchange.status != 'COMPLETE':
                raise RuntimeError('Bulkchange failed')

        bulkchange = self._startrek.bulkchange.update(self._issues, **rules)
        wait_for_bulkchange(bulkchange)


class QueryBasedRule(TransitRule):
    __metaclass__ = abc.ABCMeta

    IGNORE_COMPONENTS = set()
    QUEUE = 'BROWSER'

    def _query_components(self):
        return [
            'Queue: {}'.format(self.QUEUE),
        ] + [
            'Components: !"{}"'.format(component)
            for component in self.IGNORE_COMPONENTS
        ]

    def _filter_issues(self, issue):
        return True

    def _list_issues(self):
        issues = query_startrek(self._startrek, ' and '.join(self._query_components()))
        return filter(self._filter_issues, issues)
