import sandbox.common.errors as ce
import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.projects.browser.sb_lite.builds.TestSBLiteTask import TestSBLiteTask
from sandbox.projects.sandbox.build_sandbox_tasks import BuildSandboxTasks
from sandbox.sdk2.service_resources import SandboxTasksArchive

FINISH_STATUSES = list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK)


class RunTestTaskForCI(sdk2.Task):
    """
    Searches for tasks archive resource by arcadia commit hash and
    starts SB Lite test task with found task archive.
    """

    class Parameters(sdk2.Parameters):
        arcadia_commit_hash = sdk2.parameters.String('Arcadia commit hash')
        task_parameters = sdk2.parameters.Dict("Test task parameters")

    @classmethod
    def find_tasks_archive_task(cls, arcadia_commit_hash):
        return sdk2.Task.find(
            task_type=BuildSandboxTasks,
            input_parameters={
                'packaging': 'RAW',
                'tasks_svn_url': 'arcadia-arc:/#{}'.format(arcadia_commit_hash),
            },
        ).order(-sdk2.Task.id).first()

    @classmethod
    def find_tasks_archive_resource(cls, tasks_archive_task):
        return sdk2.Resource.find(
            task_id=tasks_archive_task.id,
            resource_type=SandboxTasksArchive,
        ).order(-sdk2.Resource.id).first()

    def on_execute(self):
        tasks_archive_task = self.find_tasks_archive_task(self.Parameters.arcadia_commit_hash)

        # Wait while tasks archive task will be started (5 min max).
        if tasks_archive_task is None:
            with self.memoize_stage.wait_for_task(max_runs=10):
                raise sdk2.WaitTime(30)
            raise ce.TaskError('Task archive task is not found')

        task_archive_task_url = 'https://sandbox.yandex-team.ru/task/{}/view'.format(tasks_archive_task.id)
        self.set_info('Task archive task: {}'.format(task_archive_task_url))

        # Wait for tasks archive task finishes.
        if tasks_archive_task.status not in ctt.Status.Group.FINISH:
            raise sdk2.WaitTask(tasks_archive_task, FINISH_STATUSES)
        if tasks_archive_task.status != ctt.Status.SUCCESS:
            raise ce.TaskFailure('Task archive task is failed')

        # Get tasks archive resource.
        tasks_archive_resource = sdk2.Resource.find(
            task_id=tasks_archive_task.id,
            resource_type=SandboxTasksArchive,
        ).first()

        task_archive_resource_url = 'https://sandbox.yandex-team.ru/resource/{}/view'.format(tasks_archive_resource.id)
        self.set_info('Task archive resource: {}'.format(task_archive_resource_url))

        # Check test task is already started and finishes.
        test_task = sdk2.Task.find(
            task_type=TestSBLiteTask,
            parent=self,
        ).order(-sdk2.Task.id).first()
        if test_task is not None:
            if test_task.status != ctt.Status.SUCCESS:
                raise ce.TaskFailure('Test task status != SUCCESS')
            else:
                self.set_info('Test task status is SUCCESS')
                return
        else:
            # Start test task.
            test_task = TestSBLiteTask(
                self,
                tasks_archive=tasks_archive_resource,
                **self.Parameters.task_parameters)
            test_task_url = 'https://sandbox.yandex-team.ru/task/{}/view'.format(test_task.id)
            self.set_info('Test task: {}'.format(test_task_url))
            test_task.save()
            test_task.enqueue()
            raise sdk2.WaitTask(test_task, FINISH_STATUSES)
