import os

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.projects.browser.common.contextmanagers import TempEnvironment
from sandbox.projects.browser.common.depot_tools import DepotToolsEnvironment
from sandbox.projects.browser.common.git import repositories, GitEnvironment
from sandbox.projects.browser.sb_lite.common import AbstractSBLiteResource
from sandbox.projects.browser.sb_lite.common import AbstractSBLiteTask
from sandbox.sandboxsdk.environments import SandboxEnvironment
from sandbox.sdk2.helpers import subprocess


class BrowserSBLiteResource(AbstractSBLiteResource):
    """
    Browser Sandbox Lite resource.
    """

    calc_md5 = False
    share = False


class BrowserBaseSBLiteTask(AbstractSBLiteTask):
    """
    Browser base Sandbox Lite task.
    """

    class Requirements(AbstractSBLiteTask.Requirements):
        dns = ctm.DnsType.DNS64

        environments = (
            GitEnvironment('2.24.1'),
        )

        class Caches(AbstractSBLiteTask.Requirements.Caches):
            pass

    class Parameters(AbstractSBLiteTask.Parameters):
        kill_timeout = 60 * 60 * 6

        with sdk2.parameters.Group('Repository') as repository_settings:
            branch = sdk2.parameters.String(
                'Git branch',
                description='Git branch to build on.')
            commit = sdk2.parameters.String(
                'Git commit',
                description='Git commit to build on.')

        abro_build_secrets_uuid = sdk2.parameters.String(
            'ABRO build secrets',
            description='YAV UUID of ABRO build secrets.',
            default='sec-01d0yk82cac106qnr30e226h20')

        arguments = sdk2.parameters.Dict(
            'Arguments',
            description='Browser build arguments.')

    def on_save(self):
        super(BrowserBaseSBLiteTask, self).on_save()

        launcher_context = self.Context.sandbox_task_launcher_context
        if launcher_context != ctm.NotExists:
            for k, v in launcher_context.iteritems():
                self.Parameters.arguments.setdefault(k, v)
        self.Parameters.arguments.setdefault('branch', self.Parameters.branch)

    def _create_worker_helper(self, browser_git_commit=None):
        """
        :type browser_git_commit: str | None
        :rtype: sandbox_lite_runner.helper.WorkerHelper
        """
        from sandbox_lite_runner.helper import WorkerHelper

        worker_type = type(self)
        if not browser_git_commit:
            browser_git_commit = self.Parameters.commit

        class BrowserWorkerHelper(WorkerHelper):
            def __init__(self, my_task):
                """
                :type my_task: BrowserBaseSBLiteTask
                """
                super(BrowserWorkerHelper, self).__init__(
                    my_task,
                    worker_type, BrowserSBLiteResource,
                    dict(my_task.Parameters),
                    'https://bitbucket.browser.yandex-team.ru/scm/stardust/browser.git',
                    str(my_task.Parameters.branch),
                    str(browser_git_commit))

            def arguments_to_parameters(self, arguments):
                return {
                    BrowserBaseSBLiteTask.Parameters.arguments.name: arguments,
                    BrowserBaseSBLiteTask.Parameters.commit.name: browser_git_commit,
                }

            def parameters_to_arguments(self, parameters):
                """
                :type parameters: BrowserBaseSBLiteTask.Parameters
                """
                return parameters.arguments

        return BrowserWorkerHelper(self)

    def _start_first_worker(self, worker_helper):
        return worker_helper.start_worker(self.Parameters.arguments)

    def browser_path(self, *args):
        return self.path('browser', *args)

    @property
    def gclient_cache_path(self):
        return SandboxEnvironment.exclusive_build_cache_dir('gclient')

    @property
    def gclient_runhooks_cache_path(self):
        return SandboxEnvironment.exclusive_build_cache_dir('gclient_hooks')

    @property
    def cipd_cache_path(self):
        return SandboxEnvironment.exclusive_build_cache_dir('cipd_cache')

    @property
    def get_deps_cache_path(self):
        return os.path.join(SandboxEnvironment.build_cache_dir, 'get-deps')

    @property
    def gradle_user_home_path(self):
        return SandboxEnvironment.exclusive_build_cache_dir('gradle')

    def gclient_sync_extra_env(self):
        return {
            'CIPD_CACHE_DIR': self.cipd_cache_path,
            'GCLIENT_CACHE_DIR': self.gclient_cache_path,
            'GIT_CACHE_PATH': self.gclient_cache_path,
            'USE_CIPD_PROXY': '1',
            'GCLIENT_RUNHOOKS_CACHE_DIR': self.gclient_runhooks_cache_path,
        }

    def gclient_sync(self):
        cmd = [
            'gclient', 'sync',
            '--nohooks',
            '--verbose',
            '--force',
            '--deps=android',
            '--ignore_locks',
        ]

        extra_env = self.gclient_sync_extra_env()
        env = dict(os.environ, **extra_env)

        with sdk2.helpers.ProcessLog(self, logger='browser_gclient_sync') as log:
            sdk2.helpers.subprocess.check_call(
                cmd, cwd=str(self.browser_path()),
                stdout=log.stdout, stderr=log.stdout, env=env)

    @classmethod
    def browser_cache_path(cls, *args):
        return os.path.join(SandboxEnvironment.exclusive_build_cache_dir('browser'), *args)

    def task_extra_env(self):
        abro_build_secrets = sdk2.yav.Secret(self.Parameters.abro_build_secrets_uuid).data()
        return dict({
            # sandbox token to start tasks with tests.
            'SANDBOX_TOKEN': abro_build_secrets['sandbox-token'],
            # token to access to yav vault.
            'YAV_TOKEN': abro_build_secrets['yav-token'],
            # sandbox task id.
            'SANDBOX_TASK_ID': str(self.Parameters.dispatcher_task_id) or '',
            # teamcity build id.
            'TEAMCITY_BUILD_ID': str(self.Parameters.teamcity_build_id) or '',
            # gradle cache dir.
            'GRADLE_USER_HOME': self.gradle_user_home_path,
            # yandex signer oauth token.
            'YANDEX_SIGNER_OAUTH': abro_build_secrets['signer-oauth-token'],

            # Gclient (sync and runhooks) variables
            'GET_DEPS_CACHE_DIR': os.path.join(SandboxEnvironment.build_cache_dir, 'get-deps'),
        }, **self.gclient_sync_extra_env())

    @property
    def _python_version(self):
        # Old browser versions do not have Python 3 support (added in 482b52878cc4ec88a4c96c343b118c17842efd02).
        # TODO: use Python 3 always when <= 22.1 versions will be obsoleted.
        python_3_support = subprocess.call(
            ['git', 'merge-base', '--is-ancestor', '482b52878cc4ec88a4c96c343b118c17842efd02', 'HEAD'],
            cwd=str(self.browser_path()),
        ) == 0
        return '3.9.7' if python_3_support else '2.7.17'

    @property
    def _pip_version(self):
        # Old browser versions do not have pip 20 support (added in eb51ebe17a6230f1ddbf93ef1921614ef3a72040).
        # TODO: use pip 20 always when <= 20.11 versions will be obsoleted.
        pip_20_support = subprocess.call(
            ['git', 'merge-base', '--is-ancestor', 'eb51ebe17a6230f1ddbf93ef1921614ef3a72040', 'HEAD'],
            cwd=str(self.browser_path()),
        ) == 0
        return '20.3.4' if pip_20_support else '9.0.1'

    def _enter_python_environment(self, exit_stack, requirements_paths):
        hpe = super(BrowserBaseSBLiteTask, self)._enter_python_environment(exit_stack, requirements_paths)

        # Add `VIRTUALENV_PYTHON` variable for backward compatibility with <= 20.9 browser versions.
        # TODO: remove this code when these versions will be obsoleted.
        if self.browser_path('src', 'build', 'yandex', 'common', 'run_build_script_with_virtualenv.py').is_file():
            temp_env = exit_stack.enter_context(TempEnvironment())
            temp_env.set_var('VIRTUALENV_PYTHON', str(hpe.python_executable))

        return hpe

    def on_execute_worker(self):
        with self._step('git clone'):
            browser_git = repositories.Stardust.browser(filter_branches=False)
            browser_git.clone(str(self.browser_path()), self.Parameters.branch, self.Parameters.commit)
            browser_git_commit = sdk2.helpers.subprocess.check_output(
                [browser_git.executable, 'rev-parse', 'HEAD'],
                cwd=str(self.browser_path())).strip()

        with self._step('prepare depot_tools'):
            DepotToolsEnvironment(
                deps_file=str(self.browser_path('src', '.DEPS.snapshot')),
                dep_name='src/third_party/depot_tools',
            ).prepare()

        depends_on_yin = not self.browser_path('src', 'build', 'yandex', 'yin_legacy', 'no_yin_dep.flag').is_file()
        if depends_on_yin:
            with self._step('gclient sync'):
                self.gclient_sync()

        requirements_paths = [
            self.browser_path('requirements.txt'),
        ]
        if depends_on_yin:
            requirements_paths.append(self.browser_path('yin', 'requirements.txt'))

        task_class = 'yandex.android.build_sandbox.BuildTask'
        env = dict(self.task_extra_env(),
                   PYTHONPATH=os.pathsep.join(['src/build']))
        resources_dir = self.path('resources')
        cache_dir = self.browser_cache_path()
        output_dir = self.path('output')
        work_dir = self.browser_path()

        worker_helper = self._create_worker_helper(browser_git_commit)
        ipc_runner = self._create_ipc_runner(
            worker_helper, resources_dir, cache_dir, output_dir, work_dir)
        self._run_task(
            ipc_runner, requirements_paths,
            task_class, env, output_dir, work_dir)
