import Queue
import contextlib
import json
import os
import textwrap
import time

import sandbox.common.errors as ce
import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects.browser.common.contextmanagers import ExitStack
from sandbox.projects.browser.common.hpe import HermeticPythonEnvironment
from sandbox.projects.browser.common.timeout import GracefulKillBeforeTimeoutMixin
from sandbox.projects.common.teamcity import TeamcityArtifacts
from sandbox.projects.common.teamcity import TeamcityArtifactsContext
from sandbox.projects.common.teamcity import TeamcityServiceMessagesLog
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2 import paths
from sandbox.sdk2.helpers import subprocess


class AbstractSBLiteResource(sdk2.Resource):
    """
    Abstract Sandbox Lite resource.
    """
    __abstract__ = True

    resource_name = sdk2.Attributes.String('Resource name', required=True)
    ttl = 1


class AbstractSBLiteTask(GracefulKillBeforeTimeoutMixin, sdk2.Task):
    """
    Abstract Sandbox Lite task.
    """

    _MODULE_PREFIX_VERSION = 'version:'
    _MODULE_VERSION_FMT = '{}=={}'
    _MODULE_PREFIX_GIT = 'git:'
    _MODULE_GIT_FMT = (
        'git+https://bitbucket.browser.yandex-team.ru/scm/'
        'mbroinfra/sandbox-integration.git@{1}#egg={0}&subdirectory={0}')

    class Parameters(sdk2.Parameters):
        teamcity_build_id = sdk2.parameters.Integer(
            'TeamCity build ID',
            description='ID of TeamCity build triggered this task.',
            required=False, default=None)
        dispatcher_task_id = sdk2.parameters.Integer(
            'Dispatcher task ID',
            description='ID of Sandbox task that dispatches this tasks hierarchy.',
            required=False, default=None)
        lxc_container = sdk2.parameters.Container(
            'LXC container',
            description='LXC container to emulate environment.',
            required=False, default=None)
        tasks_archive = sdk2.parameters.Resource(
            'Tasks archive',
            description='Resource containing tasks archive to use.',
            required=False, default=None)

        with sdk2.parameters.Group('Sandbox Lite modules') as sandbox_lite_settings:
            sandbox_lite_runner_modules = sdk2.parameters.String(
                'Sandbox Lite Runner',
                description='Sandbox Lite modules to install as runner.',
                multiline=True)
            sandbox_lite_ipc_modules = sdk2.parameters.String(
                'Sandbox Lite IPC',
                description='Sandbox Lite modules to install as IPC.',
                multiline=True)
            sandbox_lite_version = sdk2.parameters.String(
                'Sandbox Lite version',
                description=('Sandbox Lite version to generate modules list.'
                             'Supported formats are: '
                             'version:<VERSION>, git:<BRANCH>, git:<COMMIT>.'))

    def _on_save_copy_dispatcher_parameters(self):
        """
        Copies parameters to this task.
        """
        if self.parent is None:
            self.Parameters.dispatcher_task_id = self.id
        else:
            if self.Parameters.dispatcher_task_id is None:
                if isinstance(self.parent, AbstractSBLiteTask):
                    self.Parameters.dispatcher_task_id = self.parent.id
                else:
                    self.Parameters.dispatcher_task_id = self.id
        if self.Parameters.dispatcher_task_id == self.id:
            return

        dispatcher_task = AbstractSBLiteTask[self.Parameters.dispatcher_task_id]
        self.Parameters.teamcity_build_id = dispatcher_task.Parameters.teamcity_build_id
        self.Parameters.lxc_container = dispatcher_task.Parameters.lxc_container
        self.Parameters.tasks_archive = dispatcher_task.Parameters.tasks_archive
        self.Parameters.sandbox_lite_runner_modules = dispatcher_task.Parameters.sandbox_lite_runner_modules
        self.Parameters.sandbox_lite_ipc_modules = dispatcher_task.Parameters.sandbox_lite_ipc_modules
        self.Parameters.sandbox_lite_version = dispatcher_task.Parameters.sandbox_lite_version

        self.Parameters.kill_timeout = dispatcher_task.Parameters.kill_timeout

    def _on_save_task_archive(self):
        """
        Initializes task archive from parameter.
        """
        self.Requirements.tasks_resource = self.Parameters.tasks_archive

    def _on_save_sandbox_lite_modules(self):
        """
        Initializes sandbox lite modules from version.
        """
        sb_lite_version = self.Parameters.sandbox_lite_version
        if not sb_lite_version:
            return

        if sb_lite_version.startswith(self._MODULE_PREFIX_VERSION):
            version = sb_lite_version[len(self._MODULE_PREFIX_VERSION):]
            module_fmt = self._MODULE_VERSION_FMT
        elif sb_lite_version.startswith(self._MODULE_PREFIX_GIT):
            version = sb_lite_version[len(self._MODULE_PREFIX_GIT):]
            module_fmt = self._MODULE_GIT_FMT
        else:
            raise errors.TaskError(
                ('Wrong sandbox lite version: "{}". Supported formats are: '
                 'version:<VERSION>, git:<BRANCH>, git:<COMMIT>.').format(sb_lite_version))

        ipc_module = module_fmt.format('sandbox-lite-ipc', version)
        runner_module = module_fmt.format('sandbox-lite-runner', version)

        if not self.Parameters.sandbox_lite_ipc_modules:
            self.Parameters.sandbox_lite_ipc_modules = ipc_module
        if not self.Parameters.sandbox_lite_runner_modules:
            self.Parameters.sandbox_lite_runner_modules = ipc_module + '\n' + runner_module

    def on_save(self):
        self._on_save_copy_dispatcher_parameters()
        self._on_save_task_archive()
        self._on_save_sandbox_lite_modules()

    def _setup_container_environ(self):
        """
        Initializes environment in LXC-container.

        When task is executed in LXC-container it doesn't init
        environment. So we should programmatically get environment
        from /etc/profile and ~/.bashrc
        """
        # Skip initialization if container not set.
        if not self.Parameters.lxc_container:
            return

        environ_json = subprocess.check_output([
            'bash', '-c', textwrap.dedent('''
                for script in "$@"; do
                    . "$script"
                done
                python -c """
                import json
                import os
                print(json.dumps(dict(os.environ)))
                """
            '''), 'script.sh',  # Command line argument "$0".
            '/etc/profile', os.path.expanduser('~/.bashrc')])
        os.environ.clear()
        os.environ.update(json.loads(environ_json))

    def _install_sandbox_lite_runner(self):
        """
        Installs sandbox lite runner modules.
        """
        for module in self._parse_modules(self.Parameters.sandbox_lite_runner_modules):
            PipEnvironment(module).prepare()

    def is_dispatcher(self):
        """
        :return: True if this task is a dispatcher task.
        :rtype: bool
        """
        return self.Parameters.dispatcher_task_id == self.id

    def on_execute(self):
        self._setup_container_environ()
        self._install_sandbox_lite_runner()

        if self.is_dispatcher():
            self.on_execute_dispatcher()
        else:
            self.on_execute_worker()

    def _create_worker_helper(self):
        """
        :rtype: sandbox_lite_runner.helper.WorkerHelper
        """
        raise NotImplementedError()

    def _start_first_worker(self, worker_helper):
        """
        :type worker_helper: sandbox_lite_runner.helper.WorkerHelper
        :rtype: AbstractSBLiteTask
        """
        raise NotImplementedError()

    def _collect_resources(self, worker_ids):
        """
        Collects TeamCity artifacts and logs from workers.

        :type worker_ids: list[int]
        """
        collect_dir = self.path('teamcity-resources')
        collect_resources(self, collect_dir, worker_ids, TeamcityArtifacts)
        collect_resources(self, collect_dir, worker_ids, TeamcityServiceMessagesLog)

    def on_execute_dispatcher(self):
        """
        Executes as a dispatcher.
        """
        worker_helper = self._create_worker_helper()
        worker_ids = get_all_children_ids(worker_helper, self)

        with self.memoize_stage.start_worker:
            worker = self._start_first_worker(worker_helper)
            worker_ids.append(worker.id)

        statuses_to_wait = ctt.Status.Group.FINISH + ctt.Status.Group.BREAK

        if not check_all_tasks_statuses(worker_ids, statuses_to_wait):
            raise sdk2.WaitTask(tasks=worker_ids, statuses=statuses_to_wait)
        else:
            self._collect_resources(worker_ids)

            if check_any_tasks_statuses(worker_ids, ctt.Status.Group.BREAK):
                raise ce.TaskError('Some tasks broken.')
            if not check_all_tasks_statuses(worker_ids, ctt.Status.Group.SUCCEED):
                raise ce.TaskFailure('Some tasks failed.')
            return

    def _get_execution_index(self):
        """
        :return: execution index.
        :rtype: int
        """
        return self.Context.sandbox_lite_execution_index or 0

    def _create_teamcity_context(self, name=None, checkout_dir=None):
        """
        Prepares TeamCity artifacts context.

        :param name: step name or None.
        :type name: str | None
        :param checkout_dir: checkout directory or None.
        :type checkout_dir: pathlib2.Path | None
        :return: TeamCity artifacts context.
        :rtype: sandbox.projects.common.teamcity.TeamcityArtifactsContext
        """
        tc_description = 'Task #{id} {step}({time})'.format(
            id=self.id,
            step='' if not name else '[{}] '.format(name),
            time=time.strftime("%Y-%m-%d-%H-%M-%S"))
        tac = TeamcityArtifactsContext(
            base_dir=checkout_dir,
            path_suffix='{}{}'.format(
                self._get_execution_index(),
                '' if not name else '-{}'.format(name)),
            log_name=name,
            tc_service_messages_description=tc_description,
            tc_artifacts_description=tc_description + ': artifacts')
        return tac

    @contextlib.contextmanager
    def _step(self, name=None):
        """
        :type name: str | None
        """
        with self._create_teamcity_context(name) as tac:
            try:
                yield tac
            except Exception:
                tac.logger.exception('Task #{} : step "{}" failed'.format(self.id, name))
                tac.logger.info("##teamcity[buildProblem description='{}']".format(
                    'Task #{} : step "{}" failed.'.format(self.id, name)))
                raise

    @classmethod
    def _parse_modules(cls, parameter_modules):
        """
        Parses input modules parameter value and returns list of modules.

        :type parameter_modules: sdk2.parameters.String
        :rtype: list[str]
        """
        return [module for module in parameter_modules.splitlines() if module.strip()]

    @property
    def _python_version(self):
        """
        :return: version of Python.
        :rtype: str
        """
        raise NotImplementedError()

    @property
    def _pip_version(self):
        """
        :return: version of pip to install requirements.
        :rtype: str | None
        """
        return None

    def _enter_python_environment(self, exit_stack, requirements_paths):
        """
        Prepares Python environment context manager and installs passed list of
        requirements and sandbox lite IPC into it.

        :param exit_stack: stack for context managers.
        :type exit_stack: ExitStack
        :param requirements_paths: list of requirements to install.
        :type requirements_paths: list[pathlib2.Path]
        :return: Python environment.
        :rtype: sandbox.projects.browser.common.hpe.HermeticPythonEnvironment
        """
        return exit_stack.enter_context(HermeticPythonEnvironment(
            python_version=self._python_version,
            pip_version=self._pip_version,
            requirements_files=requirements_paths,
            packages=self._parse_modules(self.Parameters.sandbox_lite_ipc_modules),
        ))

    def _create_ipc_runner(self, worker_helper, resources_dir,
                           cache_dir, output_dir, work_dir):
        """
        Creates IPC runner.

        :type worker_helper: sandbox_lite_runner.helper.WorkerHelper
        :type resources_dir: pathlib2.Path
        :type cache_dir: pathlib2.Path
        :type output_dir: pathlib2.Path
        :type work_dir: pathlib2.Path
        :rtype: sandbox_lite_ipc.task_runner.IPCRunner
        """
        from sandbox_lite_ipc.task_runner import IPCRunner
        from sandbox_lite_runner.implementation import SimpleIPCMessages
        from sandbox_lite_runner.implementation import SimpleIPCReturnHandler

        ipc_messages = SimpleIPCMessages(
            self, worker_helper, str(resources_dir),
            str(cache_dir), str(output_dir), str(work_dir))
        ipc_return_handler = SimpleIPCReturnHandler()

        return IPCRunner(ipc_messages, ipc_return_handler)

    def _run_task(
            self, ipc_runner, requirements_paths,
            task_class, env, output_dir, work_dir):
        """
        Runs sandbox lite task.

        :type ipc_runner: sandbox_lite_ipc.task_runner.IPCRunner
        :type requirements_paths: list[pathlib2.Path]
        :type task_class: str
        :type env: dict[str, str]
        :type output_dir: pathlib2.Path
        :type work_dir: pathlib2.Path
        """
        with ExitStack() as exit_stack:
            with self._step('hpe'):
                hpe = self._enter_python_environment(exit_stack, requirements_paths)

            tac = self._create_teamcity_context(checkout_dir=work_dir)
            exit_stack.enter_context(tac)
            tac.logger.info('##teamcity[publishArtifacts \'{}\']'.format(output_dir))

            with ipc_runner:
                args = ipc_runner.create_args(str(hpe.python_executable), task_class)
                new_env = dict(os.environ, **env)
                sdk2.helpers.subprocess.check_call(
                    args, cwd=str(work_dir), env=new_env,
                    stdout=tac.output, stderr=subprocess.STDOUT)

            self.Context.sandbox_lite_execution_index = self._get_execution_index() + 1
            ipc_runner.handle_return()

    def on_execute_worker(self):
        """
        Executes as a worker.

        Use ``_step()``, ``_create_ipc_runner()`` and ``_run_task()`` methods
        to run sandbox lite task.
        """
        raise NotImplementedError()


def get_all_children_ids(worker_helper, sandbox_task):
    """
    Returns list of all children workers sorted by created date.

    :type worker_helper: sandbox_lite_runner.helper.WorkerHelper
    :type sandbox_task: sdk2.Task
    :rtype: list[int]
    """
    all_children_ids = []
    task_id_queue = Queue.Queue()
    task_id_queue.put(sandbox_task.id, block=False)

    while not task_id_queue.empty():
        task_id = task_id_queue.get(block=False)

        # Skip if this task is already in the list.
        if task_id in all_children_ids:
            continue

        # Add task into the list.
        all_children_ids.append(task_id)

        # Add all children tasks into the queue.
        children_ids = worker_helper.get_children_worker_ids(task_id)
        for task_id in children_ids:
            task_id_queue.put(task_id, block=False)

    # Remove first task from the list.
    all_children_ids.remove(sandbox_task.id)

    return all_children_ids


def check_all_tasks_statuses(task_ids, statuses):
    """
    :type task_ids: list[int]
    :type statuses: list[sandbox.common.types.task.Status]
    :rtype: bool
    """
    return all(sdk2.Task[task_id].status in statuses for task_id in task_ids)


def check_any_tasks_statuses(task_ids, statuses):
    """
    :type task_ids: list[int]
    :type statuses: list[sandbox.common.types.task.Status]
    :rtype: bool
    """
    return any(sdk2.Task[task_id].status in statuses for task_id in task_ids)


def collect_resources(target_task, collect_dir, source_task_ids, resource_type):
    """
    Collects resources from tasks to task.

    :type target_task: sdk2.Task
    :type collect_dir: pathlib2.Path
    :type source_task_ids: list[int]
    :type resource_type: type
    """
    resources = []
    for source_task_id in source_task_ids:
        resources.extend(find_all(
            sdk2.Resource.find, sdk2.Resource.created,
            resource_type=resource_type, task_id=source_task_id))
    resources = sorted(resources, key=lambda r: r.created)

    for resource in resources:
        resource_data = sdk2.ResourceData(resource)
        new_path = collect_dir.joinpath(str(resource.id), resource_data.path.name)
        paths.copy_path(str(resource_data.path), str(new_path))
        new_resource = resource_type(target_task, resource.description, new_path)
        sdk2.ResourceData(new_resource).ready()


__FIND_PAGE_SIZE = 1000


def find_all(find_func, order, **kwargs):
    """
    Searches for all sandbox objects using find() function with pagination.

    :param find_func: find() function.
    :param order: order() value.
    :param kwargs: arguments for find() function.
    :return: list of objects.
    """
    objects = []
    while True:
        objects_query = find_func(**kwargs).order(order)
        page_objects = list(
            objects_query.offset(len(objects)).limit(__FIND_PAGE_SIZE))
        objects.extend(page_objects)
        if len(page_objects) < __FIND_PAGE_SIZE:
            return objects
