# -*- coding: utf-8 -*-
import requests

from sandbox import sdk2
from sandbox import common
from sandbox.sandboxsdk.environments import PipEnvironment

TEAMCITY_SERVER_URL = 'teamcity.browser.yandex-team.ru'
AB_SERVER_URL = 'https://ab.yandex-team.ru'


class BrowserServerExperimentsAbTask(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 100
        environments = (
            PipEnvironment('startrek-client', '2.3', custom_parameters=['--upgrade-strategy only-if-needed']),
            PipEnvironment('teamcity-client', '4.8.2', custom_parameters=['--upgrade-strategy only-if-needed'])
        )

    class Parameters(sdk2.Parameters):
        task_id = sdk2.parameters.String("Task ID", required=True)
        state_prev = sdk2.parameters.String("Previous state", required=True)
        state_cur = sdk2.parameters.String("Current state", required=True)

        with sdk2.parameters.Group("Credentials") as credentials_group:
            robot_startrek_vault = sdk2.parameters.String(
                "Vault item with robot startrek token",
                default="robot-browser-bobr_startrek_token"
            )
            robot_teamcity_vault = sdk2.parameters.String(
                "Vault item with robot teamcity token",
                default="robot-browser-infra_teamcity_token"
            )
            robot_abt_vault = sdk2.parameters.String(
                "Vault item with abt robot token",
                default="robot-browser-bobr_abt_token"
            )

    @property
    @common.utils.singleton
    def startrek_client(self):
        import startrek_client
        return startrek_client.Startrek(
            token=sdk2.Vault.data(self.Parameters.robot_startrek_vault),
            useragent=self.__class__.name
        )

    @property
    @common.utils.singleton
    def teamcity_client(self):
        import teamcity_client
        return teamcity_client.TeamcityClient(
            server_url=TEAMCITY_SERVER_URL,
            auth=sdk2.Vault.data(self.Parameters.robot_teamcity_vault)
        )

    @property
    @common.utils.singleton
    def ab_client(self):
        abt_token = sdk2.Vault.data(self.Parameters.robot_abt_vault)
        return ABClient(AB_SERVER_URL, abt_token)

    def on_save(self):
        tags = ['BROWSER_SERVER_EXPERIMENTS']
        if len(self.Parameters.task_id) > 0:
            tags.append(self.Parameters.task_id)
        self.Parameters.tags = tags


class ABClient(object):
    class ChangeTaskStateResponse(object):
        def __init__(self, is_success, message):
            self.is_success = is_success
            self.message = message

    def __init__(self, url, abt_token):
        self.api_url = url + "/api"
        self.session = requests.session()
        headers = {
            "Authorization": "OAuth {}".format(abt_token),
            "Content-Type": "application/x-www-form-urlencoded"
        }
        self.session.headers = headers

    def get_task(self, task_id):
        resp = self.session.get(
            "{}/task/{}".format(self.api_url, task_id)
        )
        if resp.status_code == requests.codes["ok"]:
            return resp.json()
        return {"error": resp.text}

    def change_task_state(self, task_id, state):
        resp = self.session.patch(
            "{}/task/{}".format(self.api_url, task_id),
            data="state={}".format(state)
        )

        success = resp.status_code == requests.codes["ok"]
        if resp.status_code == requests.codes["ok"]:
            msg = "Статус заявки AB был успешно изменен на {}.".format(state)
        elif resp.status_code == requests.codes["forbidden"]:
            msg = ("Добавьте @robot-bobr в соавторы заявки, "
                   "откатите в черновик и отправьте на рассмотрение повторно.")
        else:
            msg = ("Невозможно изменить статус заявки AB на {}. Ошибка: {}\n"
                   "Пожалуйста, сделайте это руками.").format(state, resp.text)
        return self.ChangeTaskStateResponse(success, msg)


def sandbox_task_link(task_id):
    return "https://sandbox.yandex-team.ru/task/{}/view".format(task_id)
