import os
import json
import binascii

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types.task import Semaphores, Status
from sandbox.projects.browser.common.git import repositories, GitEnvironment, ConfigureGitEnvironment

ROBOT_BOBR_SECRET_ID = "sec-01csvvmjxg4p010h000ashpzx6"
TVM_SECRET_ID = "sec-01fqe4tvzsgbq8542x7216sg02"
TVM_TOOL_PORT = "12346"


class BrowserServerExperimentsApplyChanges(sdk2.Task):
    """Check and apply experiments change requests"""

    class Requirements(sdk2.Task.Requirements):
        platform = "linux"
        environments = [
            GitEnvironment('2.24.1'),
            ConfigureGitEnvironment('robot-bobr@yandex-team.ru', 'RobotBobr'),
        ]
        semaphores = Semaphores(
            acquires=[
                Semaphores.Acquire(name='BROWSER_SERVER_EXPERIMENTS_SYNC')
            ],
            release=[
                Status.Group.BREAK, Status.Group.FINISH
            ]
        )

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Credentials') as credentials:
            st_oauth_token = sdk2.yav.Secret(ROBOT_BOBR_SECRET_ID, default_key='startrek_token')
            ab_oauth_token = sdk2.yav.Secret(ROBOT_BOBR_SECRET_ID, default_key='ab_token')
            ssh_private_key = sdk2.yav.Secret(ROBOT_BOBR_SECRET_ID, default_key="ssh-teamcity-bitbucket-private")
            tvm_secret = sdk2.yav.Secret(TVM_SECRET_ID, default_key="client_secret")

    @staticmethod
    def _random_auth_token():
        """Generate a random string to be used as a TVM authorization token.

        More on this token:
        https://wiki.yandex-team.ru/passport/tvm2/tvm-daemon/#kakzapustitprocess
        """
        return str(binascii.hexlify(os.urandom(16)).decode("utf-8"))

    def repo_path(self, *args):
        return str(self.path('browser-uploads', *args))

    def _prepare_repo(self):
        repo = repositories.Stardust.browser_uploads(filter_branches=False)
        repo.update_cache_repo()
        repo.clone(self.repo_path(), branch="master")

    def _tvm_config_path(self):
        return str(self.path("tvm_config.json"))

    def _prepare_tvm_tool(self):
        os.environ["DEPLOY_TVM_TOOL_URL"] = "http://localhost:" + TVM_TOOL_PORT
        os.environ["TVMTOOL_LOCAL_AUTHTOKEN"] = self._random_auth_token()
        os.environ["TVM_SECRET"] = self.Parameters.tvm_secret.value(True)
        config = {
            "clients": {
                "bobr": {
                    "secret": "env:TVM_SECRET",
                    "self_tvm_id": 2032378,
                    "dsts": {
                        "JNS": {
                            "dst_id": 2029080
                        }
                    }
                }
            }
        }
        with open(self._tvm_config_path(), "w") as f:
            json.dump(config, f)
        tvmtool_resource = sdk2.Resource.find(type="TVM_TOOL_BINARY", arch=self.Requirements.platform).first()
        return str(sdk2.ResourceData(tvmtool_resource).path)

    def _prepare_bobr_tool(self):
        binary_resource = sdk2.Resource.find(type="BROWSER_BACKEND_TOOL",
                                             attrs={"name": "bobr", "platform": self.Requirements.platform})
        binary_resource = binary_resource.order(-sdk2.Resource.created).first()
        if not binary_resource:
            raise errors.TaskFailure("Could not find bobr tool binary")
        return str(sdk2.ResourceData(binary_resource).path)

    def on_execute(self):
        self._prepare_repo()
        tvm_tool_path = self._prepare_tvm_tool()
        bobr_tool_path = self._prepare_bobr_tool()
        with sdk2.helpers.ProcessLog(self, logger="tvm_tool") as tvm_log:
            tvm_tool_process = sdk2.helpers.subprocess.Popen(
                [tvm_tool_path, "-c", self._tvm_config_path(), "--port", TVM_TOOL_PORT],
                stdout=tvm_log.stdout,
                stderr=tvm_log.stderr)
            try:
                with sdk2.ssh.Key(private_part=self.Parameters.ssh_private_key.value(True)):
                    with sdk2.helpers.ProcessLog(self, logger='apply-changes') as pl:
                        sdk2.helpers.subprocess.check_call(
                            [bobr_tool_path, "experiments", "apply-changes",
                             "--auth", self.Parameters.ab_oauth_token.value(True),
                             "--st_auth", self.Parameters.st_oauth_token.value(True),
                             "--experiments-dir", self.repo_path("Experiments", "desktop")],
                            stdout=pl.stdout, stderr=pl.stderr
                        )
            finally:
                tvm_tool_process.terminate()
