# -*- coding: utf-8 -*-
import json

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Semaphores, Status
from sandbox.projects.browser.server.experiments.ab import BrowserServerExperimentsAbTask, sandbox_task_link
from sandbox.projects.browser.server.util.browser_wait_teamcity_build import BrowserServerWaitTeamcityBuild

TEAMCITY_SERVER_URL = 'teamcity.browser.yandex-team.ru'
DEPLOY_AB_EXPERIMENTS_BUILD_TYPE = 'BrowserA_BrowserUploads_UploadAbExperiments'

MASTER_STATES = ['INBOX', 'QUEUED', 'IN_CONFIG', 'RUNNING']
PRODUCTION_STATES = ['RUNNING']
ROLLBACK_STATES = ['STOPPED', 'FINISHED', 'CLOSED']

SANDBOX_TASK_LINK_TEMPLATE = "https://sandbox.yandex-team.ru/task/{}/view"
TEAMCITY_BUILD_LINK_TEMPLATE = "https://teamcity.browser.yandex-team.ru/viewLog.html?buildId={}"

EXP_IN_TESTING_TAG_NAME = "in-browser-testing"
EXP_IN_PRODUCTION_TAG_NAME = "in-browser-production"


class BrowserServerExperimentsDeploy(BrowserServerExperimentsAbTask):
    """Runs ab-experiments deploy build"""

    class Requirements(BrowserServerExperimentsAbTask.Requirements):
        semaphores = Semaphores(
            acquires=[
                Semaphores.Acquire(name='BROWSER_SERVER_EXPERIMENTS_SYNC')
            ],
            release=[
                Status.Group.BREAK, Status.Group.FINISH
            ]
        )

    class Context(sdk2.Context):
        builds = []

    @staticmethod
    def preposition(cmd):
        return "to" if cmd == "deploy" else "from"

    def status_changing_looped(self):
        tasks = list(sdk2.Task.find(self.type, tags=[self.Parameters.task_id]).order(-sdk2.Task.id).limit(2))
        if len(tasks) != 2:
            return False
        cur_task, prev_task = tasks
        return (prev_task.status == Status.FAILURE
                and cur_task.Parameters.state_prev == prev_task.Parameters.state_cur
                and cur_task.Parameters.state_cur == prev_task.Parameters.state_prev)

    def create_upload_build(self, branch, cmd):
        properties = {
            'task': self.Parameters.task_id,
            'command': cmd + "-ab",
            'prev-state': self.Parameters.state_prev,
            'next-state': self.Parameters.state_cur
        }

        environment = "production" if branch == "production" else "testing"
        description = "{cmd} {task} {preposition} {env}".format(
            cmd=cmd,
            task=self.Parameters.task_id,
            preposition=self.preposition(cmd),
            env=environment
        )
        self.Parameters.tags = self.Parameters.tags + ["{}_{}".format(cmd, environment).upper()]

        upload_build = BrowserServerWaitTeamcityBuild(
            self,
            description=description,
            notifications=self.Parameters.notifications,
            build_type=DEPLOY_AB_EXPERIMENTS_BUILD_TYPE,
            branch=branch,
            properties=json.dumps(properties)
        )
        upload_build.enqueue()
        self.Context.builds.append({"sandbox_task_id": upload_build.id, "cmd": cmd, "env": environment})
        return upload_build

    def on_success(self, prev_status):
        for build in self.Context.builds:
            self.create_startrek_build_comment(build)

    def on_failure(self, prev_status):
        for build in self.Context.builds:
            if self.status_changing_looped():
                extra = ("Перевода заявки AB в {} не будет, "
                         "чтобы не случился бесконечный цикл переходов статусов".format(prev_status))
            else:
                extra = self.ab_client.change_task_state(self.Parameters.task_id, self.Parameters.state_prev).message
            self.create_startrek_build_comment(build, extra)

    def on_break(self, prev_status, status):
        msg = "(({} Sandbox задача)) была неожиданно прервана.\n ".format(sandbox_task_link(self.id))
        change_result = self.ab_client.change_task_state(self.Parameters.task_id, self.Parameters.state_prev)
        if change_result.is_success:
            msg += "Заявка AB переведена в предыдущий статус {}".format(self.Parameters.state_prev)
        else:
            msg += ("Не удалось перевести заявку AB в предыдущий статус {state}:\n"
                    "* {error}\n".format(state=self.Parameters.state_prev, error=change_result.message))
        self.create_startrek_comment(msg)

    def on_execute(self):
        with self.memoize_stage.create_children:
            builds = []
            ab_task = self.ab_client.get_task(self.Parameters.task_id)
            decision = ab_task.get("decision", "")
            left_experiment_in_repository = False
            if self.Parameters.state_prev not in MASTER_STATES and self.Parameters.state_cur in MASTER_STATES:
                builds.append(self.create_upload_build("master", "deploy"))
            elif self.Parameters.state_prev in MASTER_STATES and self.Parameters.state_cur in ROLLBACK_STATES:
                if self.Parameters.state_cur == "CLOSED" and decision == "deploy":
                    left_experiment_in_repository = True
                else:
                    builds.append(self.create_upload_build("master", "rollback"))

            if self.Parameters.state_prev not in PRODUCTION_STATES and self.Parameters.state_cur in PRODUCTION_STATES:
                builds.append(self.create_upload_build("production", "deploy"))
            elif self.Parameters.state_prev in PRODUCTION_STATES and self.Parameters.state_cur in ROLLBACK_STATES:
                if self.Parameters.state_cur == "CLOSED" and decision == "deploy":
                    left_experiment_in_repository = True
                else:
                    builds.append(self.create_upload_build("production", "rollback"))

            if left_experiment_in_repository:
                msg = ("Заявка AB была закрыта, но эксперимент не был удалён из репозитория, "
                       "потому что ожидается раскатка на 100%")
                self.create_startrek_comment(msg)

            if len(builds) > 0:
                raise sdk2.WaitTask(
                    builds,
                    list(Status.Group.FINISH + Status.Group.BREAK),
                    wait_all=True,
                )

        for task in list(self.find()):
            for build in self.Context.builds:
                if task.id == build["sandbox_task_id"]:
                    build["sandbox_task_status"] = task.status
                    build["teamcity_build_id"] = task.Context.build_id

        for build in self.Context.builds:
            if build["sandbox_task_status"] != Status.SUCCESS:
                raise TaskFailure("Subtask {} failed".format(build["sandbox_task_id"]))

    def create_startrek_build_comment(self, build, extra=""):
        header = "Переход из {prev} в {cur} ".format(prev=self.Parameters.state_prev, cur=self.Parameters.state_cur)
        header += "!!(green)удался!!" if build["sandbox_task_status"] == Status.SUCCESS else "!!не удался!!, ошибки:"

        msgs = [header]
        if build["sandbox_task_status"] != Status.SUCCESS:
            build_problems = list(self.teamcity_client.BuildProblems(build__id=build["teamcity_build_id"]))
            msgs.extend(["* " + problem.details for problem in build_problems])
            if len(build_problems) > 0:
                msgs.append("\n")

        task = self.ab_client.get_task(self.Parameters.task_id)
        if task.get("error"):
            msgs.append("Не удалось получить информацию о заявке AB: {}\n"
                        "Проверьте, пожалуйста, статус нахождения эксперимента в репозитории самостоятельно".
                        format(task.get("error")))
        else:
            tags = task.get("tags", [])
            in_testing = EXP_IN_TESTING_TAG_NAME in tags
            in_production = EXP_IN_PRODUCTION_TAG_NAME in tags
            msgs.append("Статус в браузере: testing - {in_testing_status}, production - {in_production_status}.".format(
                in_testing_status="!!(green)да!!" if in_testing else "!!нет!!",
                in_production_status="!!(green)да!!" if in_production else "!!нет!!")
            )

        if extra:
            msgs.append(extra)

        msgs.append("(({teamcity_build_link} TeamCity build)), (({sandbox_task_link} Sandbox task))".format(
            teamcity_build_link=TEAMCITY_BUILD_LINK_TEMPLATE.format(build["teamcity_build_id"]),
            sandbox_task_link=SANDBOX_TASK_LINK_TEMPLATE.format(build["sandbox_task_id"]))
        )

        self.create_startrek_comment("\n".join(msgs))

    def create_startrek_comment(self, msg):
        issue = self.startrek_client.issues[self.Parameters.task_id]
        issue.comments.create(text=msg)
