# -*- coding: utf-8 -*-
import json
import re

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Status
from sandbox.projects.browser.server.experiments.ab import BrowserServerExperimentsAbTask, sandbox_task_link
from sandbox.projects.browser.server.util.browser_wait_teamcity_build import BrowserServerWaitTeamcityBuild

VALIDATE_AB_ABRO_EXPERIMENTS_BUILD_TYPE = 'BrowserA_BrowserUploads_ValidateAbAbroExperiments'


class BrowserServerExperimentsValidateAbro(BrowserServerExperimentsAbTask):
    """Run ab-experiments validation build for abro experiments"""

    class Context(sdk2.Context):
        build_id = None
        build_url = None

    def _create_validate_task(self):
        properties = {
            'task': self.Parameters.task_id,
        }
        validate_build = BrowserServerWaitTeamcityBuild(
            self,
            description="Run validation of abro-experiment {}".format(self.Parameters.task_id),
            notifications=self.Parameters.notifications,
            build_type=VALIDATE_AB_ABRO_EXPERIMENTS_BUILD_TYPE,
            branch="master",
            properties=json.dumps(properties)
        )
        return validate_build

    def _build_status_message(self, status, color):
        return "(({task_link} Проверка)) на дубликаты - !!({color}){status}!!.\n".format(
            task_link=sandbox_task_link(self.id),
            color=color,
            status=status
        )

    def _duplicate_message(self):
        if self.Context.build_id is None:
            return ""
        tc_build = self.teamcity_client.Build(id=self.Context.build_id)
        if tc_build.status.lower() == "success":
            return ""
        duplicate_error_message = tc_build.get_resulting_properties()["duplicate-error-message"]
        if len(duplicate_error_message) > 0:
            if not duplicate_error_message.endswith('\n'):
                duplicate_error_message += '\n'
            return re.sub(r"'(https://bitbucket.browser.yandex-team.ru.*)'",
                          r"((\g<1> bitbucket))",
                          duplicate_error_message)
        return "Посмотреть детали можно (({} здесь)).\n".format(self.Context.build_url)

    def on_success(self, prev_status):
        issue = self.startrek_client.issues[self.Parameters.task_id]
        issue.comments.create(text=self._build_status_message("OK", "green"))

    def _on_fail(self, status):
        change_result = self.ab_client.change_task_state(self.Parameters.task_id, self.Parameters.state_prev)
        issue = self.startrek_client.issues[self.Parameters.task_id]
        comment = "{status_message}\n{duplicate_message}\n{switch_message}".format(
            status_message=self._build_status_message(status, "red"),
            duplicate_message=self._duplicate_message(),
            switch_message=change_result.message
        )
        issue.comments.create(text=comment)

    def on_failure(self, prev_status):
        self._on_fail("FAILED")

    def on_break(self, prev_status, status):
        self._on_fail(status)

    def on_execute(self):
        with self.memoize_stage.create_child_task:
            validate_task = self._create_validate_task()
            validate_task.enqueue()
            raise sdk2.WaitTask(
                validate_task,
                list(Status.Group.FINISH + Status.Group.BREAK),
                wait_all=True,
            )

        task = self.find().first()
        if task is None:
            self.set_info("Child task not found")
            return

        self.Context.build_id = task.Context.build_id
        self.Context.build_url = task.Context.web_url
        if task.status != Status.SUCCESS:
            raise TaskFailure("Subtask {} failed".format(task.id))
