import json
import logging
import requests
import socket
import textwrap

from sandbox import sdk2
from sandbox import common
from sandbox.common.errors import TaskFailure
import sandbox.common.types.client as ctc
from sandbox.projects.common import decorators
from sandbox.sandboxsdk.environments import PipEnvironment


TEAMCITY_SERVER = 'teamcity.browser.yandex-team.ru'

STATUS_CSS_CLASS = {
    None: 'status_draft',
    'queued': 'status_wait_time',
    'running': 'status_executing',
    'success': 'status_success',
    'failure': 'status_exception',
    'unknown': 'status_draft',
    'unknown_finished': 'status_exception'
}


class BrowserServerWaitTeamcityBuild(sdk2.Task):
    """Runs browser teamcity server build and waits for it"""

    def finished(self):
        return self.Context.status in (
            "success",
            "failure",
            "unknown_finished"
        )

    def update_status(self, build):
        if self.finished():
            return

        if build.state == 'queued' or build.state == 'running':
            self.Context.status = build.state
        elif build.state == 'finished':
            build_status = build.status.lower()
            if build_status == 'success' or build_status == 'failure':
                self.Context.status = build_status
            elif build_status == 'unknown':
                self.Context.status = 'unknown_finished'
            else:
                logging.warning('Unknown build status for %s: %s', build.web_url, build.status)
                self.Context.status = 'unknown_finished'
        else:
            logging.warn("Unknown build state for %s: %s", build.web_url, build.state)
            self.Context.status = "unknown"

    class Requirements(sdk2.Requirements):
        kill_timeout = 3 * 60 * 60
        disk_space = 100  # MB
        cores = 1
        client_tags = ctc.Tag.BROWSER
        environments = [
            PipEnvironment('teamcity-client==4.8.2')
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        build_type = sdk2.parameters.String('Build type', required=True)
        branch = sdk2.parameters.String('Branch', default='master')
        properties = sdk2.parameters.String('Properties', default='{}')
        put_to_top = sdk2.parameters.Bool("Put to top", default=False)

        with sdk2.parameters.Group("Timeouts") as timeout_group:
            sleep_time = sdk2.parameters.Integer("Sleep time (minutes)", default=1)

        with sdk2.parameters.Group("Credentials") as credentials_group:
            robot_oauth_vault = sdk2.parameters.String(
                "Vault item with token for teamcity",
                default="robot-browser-infra_teamcity_token"
            )

    class Context(sdk2.Context):
        build_id = None
        web_url = None
        status = None

    @property
    @common.utils.singleton
    def teamcity_client(self):
        import teamcity_client.client
        return teamcity_client.client.TeamcityClient(
            server_url=TEAMCITY_SERVER,
            auth=sdk2.Vault.data(self.Parameters.robot_oauth_vault)
        )

    def trigger_build(self):
        properties = json.loads(self.Parameters.properties)
        build = self.teamcity_client.BuildType(id=self.Parameters.build_type).run_build(
            put_to_top=self.Parameters.put_to_top,
            branch=self.Parameters.branch,
            comment='{}. Triggered by {} from sandbox-task sandbox-{}'.format(
                self.Parameters.description, self.author, self.id
            ),
            properties=properties
        )
        logging.info('TeamCity Build %s started.', build.web_url)
        self.set_info('TeamCity Build {} started.'.format(build.web_url))
        return build

    @decorators.retries(5, delay=0.5, backoff=4, max_delay=512, exceptions=(requests.RequestException, socket.error))
    def load_build(self):
        if self.Context.build_id is None:
            build = self.trigger_build()
            self.Context.build_id = build.id
            self.Context.web_url = build.web_url
            return build
        else:
            return self.teamcity_client.Build(id=self.Context.build_id)

    def on_execute(self):
        build = self.load_build()
        self.update_status(build)

        if not self.finished():
            raise sdk2.WaitTime(self.Parameters.sleep_time * 60)

        self.set_info("Task finished with '{}' status".format(build.status))
        if self.Context.status != "success":
            raise TaskFailure("Build failed with status '{}'".format(build.status))

    @sdk2.header()
    def header(self):
        return textwrap.dedent(u"""
        <table>
            <tr>
                <th>Build Type </th>
                <th>Build ID </th>
                <th>Status </th>
            </tr>
            <tr>
                <td>{}</td>
                <td style="text-align: center">
                    <a href='{}'>{}</a>
                </td>
                <td style="text-align: center"><span class='status {}'>{}</span></td>
            </tr>
        </table>
        """.format(
            self.Parameters.build_type,
            self.Context.web_url,
            self.Context.build_id,
            STATUS_CSS_CLASS[self.Context.status],
            self.Context.status
        )).strip()
