# -*- coding: utf-8 -*-
import datetime
import os

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common import utils
from sandbox.sandboxsdk.environments import PipEnvironment

# Statface report contains info for 78% users only. So use this coefficient to make figures more accurate.
USERS_PERCENT_STAT_FACTOR = 1 / 0.78
# bar-navig-log contains info for 75% users only. So use this coefficient to make figures more accurate.
USERS_PERCENT_BARNAVIG_FACTOR = 1 / 0.75
# bar-navig-log.6p is 6% extraction from bar-navig-log.
USERS_PERCENT_BARNAVIG_6P_FACTOR = USERS_PERCENT_BARNAVIG_FACTOR * (1 / 0.06)

CONDITIONS_SCHEMA = {
    'type': 'array',
    'items': {
        'type': 'array',
        'items': {'$ref': '#/definitions/condition'},
        'minItems': 1,
    },
    'minItems': 1,
    'definitions': {
        'interval': {
            'type': 'array',
            'items': [
                {'type': ['integer', 'null']},
                {'type': ['integer', 'null']},
            ],
            'minItems': 2,
            'maxItems': 2,
        },
        'string_or_interval': {
            'anyOf': [
                {'type': 'string'},
                {'$ref': '#/definitions/interval'},
            ],
        },
        'titled_string': {
            'type': 'object',
            'properties': {
                'field_title': {'type': 'string'},
                'field_value': {'type': 'string'},
            },
            'required': ['field_title', 'field_value'],
        },
        'titled_string_or_interval': {
            'type': 'object',
            'properties': {
                'field_title': {'type': 'string'},
                'field_value': {'$ref': '#/definitions/string_or_interval'},
            },
            'required': ['field_title', 'field_value'],
        },
        'condition': {
            'type': 'object',
            'properties': {
                'component': {'$ref': '#/definitions/titled_string'},
                'parameter': {'$ref': '#/definitions/titled_string'},
                'value': {'$ref': '#/definitions/titled_string_or_interval'},
                'value2': {'$ref': '#/definitions/titled_string_or_interval'},
                'value3': {'$ref': '#/definitions/titled_string_or_interval'},
            },
            'dependencies': {
                'value2': {
                    'properties': {'value': {'$ref': '#/definitions/titled_string'}},
                    'required': ['value'],
                },
                'value3': {
                    'properties': {'value2': {'$ref': '#/definitions/titled_string'}},
                    'required': ['value2'],
                },
            },
            'required': ['component', 'parameter'],
            'additionalProperties': False,
        },
    },
}

# Values are IDs of `bugsVisibility` value in Startrek.
VISIBILITY_CHOICES = {
    'invisible': 535,
    'visible': 536,
    'disturbing': 537,
}
FACTORS_SCHEMA = {
    'type': 'array',
    'items': {
        'type': 'object',
        'properties': {
            'os': {'$ref': '#/definitions/titled_number'},
            'os_scale': {'$ref': '#/definitions/titled_number'},
            'additional_steps': {'$ref': '#/definitions/titled_number'},
            'deployment_percentage': {'$ref': '#/definitions/titled_number'},
            'visual': {'$ref': '#/definitions/titled_number'},
            'visibility': {'$ref': '#/definitions/titled_visibility'},
        },
        'required': ['visibility'],
        'additionalProperties': False,
    },
    'minItems': 1,
    'definitions': {
        'titled_number': {
            'type': 'object',
            'properties': {
                'field_title': {'type': 'string'},
                'field_value': {'type': 'number'},
            },
            'required': ['field_title', 'field_value']
        },
        'titled_visibility': {
            'type': 'object',
            'properties': {
                'field_title': {'type': 'string'},
                'field_value': {'enum': VISIBILITY_CHOICES.keys()},
            },
            'required': ['field_title', 'field_value']
        },
    },
}

STAT_PIP_ENVIRONMENTS = (
    PipEnvironment('python-statface-client', '0.150.0',
                   custom_parameters=['--upgrade-strategy', 'only-if-needed']),
)
STAT_REPORT_NAME = 'Chromium/Adhoc/BrowserFeatures/client_id_country_ver'

YQL_PIP_ENVIRONMENTS = (
    PipEnvironment('yandex-yt', '0.9.17'),  # `yql` depends on `yandex-yt` but does not install it.
    PipEnvironment('yql', '1.2.86'),
)
YQL_QUERY_SYNTAX_VERSION = 1


class BrowserSeverityCalculateDau(sdk2.Task):
    class Requirements(sdk2.Requirements):
        disk_space = 1 * 1024
        cores = 1
        environments = (
            PipEnvironment('pyrsistent', '0.15.7'),  # via jsonschema
            PipEnvironment('jsonschema', '3.0.2', custom_parameters=['--upgrade-strategy', 'only-if-needed']),

            PipEnvironment('requests', '2.18.4'),  # via startrek-client & python-statface-client
            PipEnvironment('startrek-client', '2.3', custom_parameters=['--upgrade-strategy', 'only-if-needed']),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        conditions = sdk2.parameters.JSON(
            'Conditions: array of objects with "component", "parameter", "value", "value2", "value3" fields',
            default=[])
        brand_id = sdk2.parameters.String('Brand ID')
        date = sdk2.parameters.String('Date (yyyy-mm-dd)', required=True)
        issue = sdk2.parameters.String('Issue for report')
        factors = sdk2.parameters.JSON('Factors', default=[])

        allow_stat = sdk2.parameters.Bool('Allow to use Stat', default=True)
        use_6p_log = sdk2.parameters.Bool('Use 6% extraction from bar-navig-log (only for YQL usage)')

        with sdk2.parameters.Group('Credentials') as credentials_group:
            stat_token_vault = sdk2.parameters.String('Vault item with Stat token',
                                                      default='robot-bro-severity_stat_token')
            startrek_token_vault = sdk2.parameters.String('Vault item with Startrek token',
                                                          default='robot-bro-severity_startrek_token')
            yql_token_vault = sdk2.parameters.String('Vault item with YQL token',
                                                     default='robot-bro-severity_yql_token')

    class Context(sdk2.Context):
        yql_operation_id = None

    def should_use_stat(self):
        import jsonschema
        return self.Parameters.allow_stat and jsonschema.Draft7Validator({
            'items': [{
                'properties': {
                    'value': {'anyOf': [{'type': 'string'}, {'type': 'object'}]},
                    'value2': {'anyOf': [{'type': 'string'}, {'type': 'object'}]},
                    'value3': {'anyOf': [{'type': 'string'}, {'type': 'object'}]},
                },
            }],
            'maxItems': 1,
        }).is_valid(self.Parameters.conditions[0]) and len(self.Parameters.conditions) == 1

    def on_prepare(self):
        super(BrowserSeverityCalculateDau, self).on_prepare()
        self.validate_parameters()
        extra_environments = STAT_PIP_ENVIRONMENTS if self.should_use_stat() else YQL_PIP_ENVIRONMENTS
        for environment in extra_environments:
            environment.prepare()

    @utils.singleton_property
    def startrek_client(self):
        import startrek_client
        return startrek_client.Startrek(
            'BROWSER_SEVERITY_CALCULATE_DAU', sdk2.Vault.data(self.Parameters.startrek_token_vault),
            headers={'Accept-Language': 'en-US, en'}, timeout=10)

    @utils.singleton_property
    def statface_client(self):
        import statface_client
        return statface_client.ProductionStatfaceClient(oauth_token=sdk2.Vault.data(self.Parameters.stat_token_vault))

    @utils.singleton_property
    def yql_client(self):
        from yql.api.v1.client import YqlClient
        return YqlClient(token=sdk2.Vault.data(self.Parameters.yql_token_vault))

    @property
    def date(self):
        return datetime.datetime.strptime(self.Parameters.date, '%Y-%m-%d')

    def validate_parameters(self):
        import jsonschema

        jsonschema.validate(self.Parameters.conditions, CONDITIONS_SCHEMA)
        jsonschema.validate(self.Parameters.factors, FACTORS_SCHEMA)
        try:
            self.date
        except ValueError:
            raise errors.TaskError('Wrong date format: {}'.format(self.Parameters.date))

    @classmethod
    def get_field_title_and_value(cls, stat_dict, key):
        if key not in stat_dict:
            return None, None
        else:
            return stat_dict[key]['field_title'], stat_dict[key]['field_value']

    @classmethod
    def get_field_title(cls, stat_dict, key):
        return cls.get_field_title_and_value(stat_dict, key)[0]

    @classmethod
    def get_field_value(cls, stat_dict, key):
        return cls.get_field_title_and_value(stat_dict, key)[1]

    @classmethod
    def get_factor_and_prefix(cls, stat_dict, key, default, type_name):
        field_title, field_value = cls.get_field_title_and_value(stat_dict, key)
        '{}{}: '.format(type_name, ' ({})'.format(field_title) if field_title else '')
        return field_value or default, '{}{}: '.format(
            type_name, ' ({})'.format(field_title) if field_title else '')

    def get_query(self):
        with open(os.path.join(os.path.dirname(os.path.abspath(__file__)), 'yql_script.py'), 'rb') as script_file:
            script = script_file.read()
        with open(os.path.join(os.path.dirname(os.path.abspath(__file__)), 'query.yql'), 'rb') as query_file:
            return query_file.read().replace('<<script>>', script)

    def get_query_parameters(self):
        from yql.client.parameter_value_builder import YqlParameterValueBuilder as Builder

        def make_string_or_interval(value):
            if isinstance(value, basestring):
                return Builder.make_variant(1, Builder.make_string(value))
            else:
                return Builder.make_variant(0, Builder.make_tuple([
                    Builder.make_null() if item is None else Builder.make_int64(item)
                    for item in value
                ]))

        def make_optional_string_or_interval(value):
            # Optional<Variant<...>> takes null or 1-tuple that contains variant.
            if value is None:
                return Builder.make_null()
            else:
                return Builder.make_tuple([make_string_or_interval(value)])

        return Builder.build_json_map({
            '$table_name': Builder.make_string('statbox/{}/{}'.format(
                'bar-navig-log.6p' if self.Parameters.use_6p_log else 'bar-navig-log',
                self.date.strftime('%Y-%m-%d'),
            )),
            '$conditions': Builder.make_list([
                Builder.make_list([
                    Builder.make_tuple([
                        Builder.make_string(self.get_field_value(condition, 'component')),
                        Builder.make_string(self.get_field_value(condition, 'parameter')),
                        make_optional_string_or_interval(self.get_field_value(condition, 'value')),
                        make_optional_string_or_interval(self.get_field_value(condition, 'value2')),
                        make_optional_string_or_interval(self.get_field_value(condition, 'value3')),
                    ]) for condition in conjunction
                ]) for conjunction in self.Parameters.conditions
            ]),
            '$brand_id': (
                Builder.make_string(self.Parameters.brand_id)
                if self.Parameters.brand_id
                else Builder.make_null()
            ),
        })

    def calculate_dau_via_stat(self):
        import statface_client

        condition = self.Parameters.conditions[0][0]

        path = ['_total_', '_total_',
                self.get_field_value(condition, 'component'),
                self.get_field_value(condition, 'parameter')]
        for key in ('value', 'value2', 'value3'):
            value = self.get_field_value(condition, key)
            if value:
                path.append(value)
        path = '\t'.join([''] + path + [''])

        report = self.statface_client.get_report(STAT_REPORT_NAME)
        data = report.download_data(
            scale=statface_client.constants.DAILY_SCALE,
            date_min=self.date.strftime('%Y-%m-%d %H:%M:%S'),
            date_max=self.date.replace(hour=23, minute=59, second=59).strftime('%Y-%m-%d %H:%M:%S'),
            path=path,
            country=self.Parameters.brand_id or '_total_',
            path__mode='subtree',
        )
        result = next((item for item in data if item['path'] == path), {})
        return int(result.get('uids', 0))

    def calculate_dau_via_yql(self):
        request = self.yql_client.query(self.get_query(), syntax_version=YQL_QUERY_SYNTAX_VERSION)
        request.run(parameters=self.get_query_parameters())
        self.Context.yql_operation_id = request.operation_id
        self.Context.save()

        result = request.get_results()
        if not request.is_success:
            raise errors.TaskError('YQL request failed')

        return int(list(result)[0].rows[0][0])

    @utils.singleton
    def st_visibility_value(self, key):
        return self.startrek_client.translations[VISIBILITY_CHOICES[key]]

    def apply_factors_combination(self, dau, combination):
        """
        :param dau: initial DAU
        :param combination: factors combination to apply
        :return: tuple of:
                 - new DAU;
                 - severity explanation.

        :type dau: int
        :type combination: dict[str, float]
        :rtype: (int, unicode)
        """
        if self.should_use_stat():
            users_percent_factor = USERS_PERCENT_STAT_FACTOR
            users_percent_factor_prefix = u'stat: '
        elif self.Parameters.use_6p_log:
            users_percent_factor = USERS_PERCENT_BARNAVIG_6P_FACTOR
            users_percent_factor_prefix = u'barnavig 6%: '
        else:
            users_percent_factor = USERS_PERCENT_BARNAVIG_FACTOR
            users_percent_factor_prefix = u'barnavig: '

        lines = []

        for factor, prefix in (
                (users_percent_factor, users_percent_factor_prefix),
                self.get_factor_and_prefix(combination, 'os', 1, u'ОС'),
                self.get_factor_and_prefix(combination, 'os_scale', 1, u'Масштабирование'),
                self.get_factor_and_prefix(combination, 'additional_steps', 1, u'Дополнительные шаги'),
                self.get_factor_and_prefix(combination, 'deployment_percentage', 1, u'Раскатка фичи'),
                self.get_factor_and_prefix(combination, 'visual', 1, u'Неочевидный визуальный баг'),
        ):
            if factor != 1:
                new_dau = int(dau * factor)
                factor_str = '{:.5f}'.format(factor).rstrip('0')
                lines.append(u'{}{} * {} = {}'.format(prefix, dau, factor_str, new_dau))
                dau = new_dau

        visibility_name = self.get_field_value(combination, 'visibility')
        visibility = self.st_visibility_value(visibility_name)
        lines.append(u'Аудитория бага: [{dau}] + [{visibility}] = [{severity}]'.format(
            dau=dau,
            visibility=visibility.value['ru'],
            severity=self.get_severity(dau, visibility),
        ))

        return dau, u'\n'.join(lines)

    def generate_condition_explanation(self):
        explanations = []
        for conjunction in self.Parameters.conditions:
            explanations.append(self.generate_conjunction_explanation(conjunction))
        return '\nOR\n'.join(explanations)

    def generate_conjunction_explanation(self, conjunction):
        explanations = []
        for condition in conjunction:
            titles = [
                self.get_field_title(condition, 'component'),
                self.get_field_title(condition, 'parameter'),
                self.get_field_title(condition, 'value'),
                self.get_field_title(condition, 'value2'),
                self.get_field_title(condition, 'value3'),
            ]
            explanations.append(' -> '.join(filter(None, titles)))
        # If conditions passed without titles - strip explanation to return empty string
        return '\n'.join(explanations).strip()

    def apply_factors(self, dau):
        """
        :param dau: initial DAU
        :return: tuple of:
                 - new DAU;
                 - severity explanation that should be written to task info;
                 - ...                                         to "Severity Explanation" field of issue;
                 - ...                                         to issue comment.

        :type dau: float
        :rtype: (int, unicode, unicode, unicode or NoneType)
        """
        condition_explanation = self.generate_condition_explanation()

        general_combination = self.Parameters.factors[0]
        general_dau, general_explanation = self.apply_factors_combination(dau, general_combination)
        if condition_explanation:
            general_explanation = condition_explanation + '\n' + general_explanation
        explanation_field = general_explanation
        if len(self.Parameters.factors) > 1:
            complete_explanation = u'\n\n'.join(
                self.apply_factors_combination(dau, combination)[1] for combination in self.Parameters.factors)
            if condition_explanation:
                complete_explanation = condition_explanation + '\n' + complete_explanation
            task_info = explanation_comment = complete_explanation
        else:
            task_info = general_explanation
            explanation_comment = None
        return general_dau, task_info, explanation_field, explanation_comment

    def get_severity(self, dau, visibility):
        if dau < 1000:
            return {
                VISIBILITY_CHOICES['invisible']: 'trivial',
                VISIBILITY_CHOICES['visible']: 'minor',
                VISIBILITY_CHOICES['disturbing']: 'normal',
            }[visibility.id]
        elif dau <= 50000:
            return {
                VISIBILITY_CHOICES['invisible']: 'minor',
                VISIBILITY_CHOICES['visible']: 'normal',
                VISIBILITY_CHOICES['disturbing']: 'critical',
            }[visibility.id]
        else:
            return {
                VISIBILITY_CHOICES['invisible']: 'normal',
                VISIBILITY_CHOICES['visible']: 'critical',
                VISIBILITY_CHOICES['disturbing']: 'blocker',
            }[visibility.id]

    def on_execute(self):
        dau = self.calculate_dau_via_stat() if self.should_use_stat() else self.calculate_dau_via_yql()

        dau, task_info, explanation_field, explanation_comment = self.apply_factors(dau)
        self.set_info(task_info)

        if self.Parameters.issue:
            visibility = self.st_visibility_value(
                self.get_field_value(self.Parameters.factors[0], 'visibility'))

            issue = self.startrek_client.issues[self.Parameters.issue]
            issue.update(
                bugsAudience=dau,
                bugsVisibility=visibility,
                severity=self.get_severity(dau, visibility),
                severityExplanation=explanation_field,
            )
            if explanation_comment:
                issue.comments.create(text=explanation_comment)

    def on_break(self, prev_status, status):
        super(BrowserSeverityCalculateDau, self).on_break(prev_status, status)
        if self.Context.yql_operation_id:
            for environment in YQL_PIP_ENVIRONMENTS:
                environment.prepare()
            from yql.client.operation import YqlAbortOperationRequest

            self.yql_client  # Strange way to authenticate.
            YqlAbortOperationRequest(self.Context.yql_operation_id).run()
