import json


def _safe_get(data, key):
    return data.get(key) if isinstance(data, dict) else None


def _value_in_interval(value, interval):
    if not isinstance(value, (int, float)):
        return False
    left = interval[0] if interval[0] is not None else -float('inf')
    right = interval[1] if interval[1] is not None else float('inf')
    return left <= value <= right


def _check_params_dict(item, conditions):
    """
    :type item: dict
    :type conditions: list[(str, str or (int, int)) or NoneType]
    :rtype: bool
    """
    conditions = list(filter(None, conditions))
    if not conditions:
        return True

    for value_type, value in conditions[:-1]:
        assert value_type == 'string'
        item = _safe_get(item, value)

    value_type, value = conditions[-1]
    if value_type == 'string':
        if isinstance(item, dict):
            return value in item
        else:
            return item == value
    else:
        assert value_type == 'interval'
        return _value_in_interval(item, value)


def _check_item(item, component, parameter, value, value2, value3):
    """
    Check that `item` fulfils condition specified by (`component`, `parameter`, `value`, `value2`, `value3`) tuple.

    :param item: dict from decoded bundle
    :param component: component name (client ID)
    :param parameter: parameter name
    :param value: tuple of value type ('string' or 'interval') and value
    :param value2: tuple of value2 type ('string' or 'interval') and value2
    :param value3: tuple of value3 type ('string' or 'interval') and value3
    :return: if condition is true

    :type item: dict
    :type component: str
    :type parameter: str
    :type value: (str, str or (int, int)) or NoneType
    :type value2: (str, str or (int, int)) or NoneType
    :type value3: (str, str or (int, int)) or NoneType
    :rtype: bool
    """
    if _safe_get(item, 'client_id') != component:
        return False
    params_dict = _safe_get(item, 'params')
    return _check_params_dict(params_dict, [('string', parameter), value, value2, value3])


def contains_conditions(decoded_bundle, conditions):
    try:
        data = json.loads(decoded_bundle or '[]')
    except ValueError:
        return False
    return any(all(
        any(_check_item(item, *condition) for item in data)
        for condition in conjunction
    ) for conjunction in conditions)
