"""
Author: Anatoly Matyukhin <amatyukhin@yandex-team.ru>
"""
import logging
import shutil
import tarfile

import sandbox.common.types.misc as ctm

from sandbox.projects.browser.common.git import BrowserGitEnvironment, GitEnvironment

from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessLog, subprocess

GIT_SOURCE_URL = 'https://github.com/git/git.git'
BIN_WRAPPER_TEMPLATE = """\
#!/usr/bin/env bash
WRAPPERS_PATH=$(cd $(dirname "$0") && pwd)
GIT_EXEC_PATH=$(cd $WRAPPERS_PATH/.. && pwd)
if test -n "$NO_SET_GIT_TEMPLATE_DIR"
then
    unset GIT_TEMPLATE_DIR
else
    GIT_TEMPLATE_DIR=$GIT_EXEC_PATH'/templates/blt'
    export GIT_TEMPLATE_DIR
fi
GITPERLLIB=$GIT_EXEC_PATH'/perl/blib/lib'"${{GITPERLLIB:+:$GITPERLLIB}}"
GIT_TEXTDOMAINDIR=$GIT_EXEC_PATH'/po/build/locale'
PATH=$GIT_EXEC_PATH'/bin-wrappers:'"$PATH"
export GIT_EXEC_PATH GITPERLLIB PATH GIT_TEXTDOMAINDIR
exec "${{GIT_EXEC_PATH}}/{filename}" "$@"
"""


class BrowserBuildGit(sdk2.Task):
    class Requirements(sdk2.Requirements):
        environments = (
            GitEnvironment('2.24.1'),
        )
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Parameters):
        commit = sdk2.parameters.String('Commit of git.git for build')
        git_version = sdk2.parameters.String('Git version')
        make_options = sdk2.parameters.Dict('Make options')
        _container = sdk2.parameters.Container('LXC container', default=None, required=False)

    def repo_path(self):
        return self.path('git')

    def clone_git(self):
        subprocess.check_call(['git', 'clone', GIT_SOURCE_URL, str(self.repo_path())])
        if self.Parameters.commit:
            subprocess.check_call(['git', 'checkout', self.Parameters.commit], cwd=str(self.repo_path()))

    def build_git(self):
        with ProcessLog(self, logger='make') as log:
            subprocess.check_call(
                ['make'] + ['{}={}'.format(k, v) for k, v in self.Parameters.make_options.iteritems()],
                cwd=str(self.repo_path()), stdout=log.stdout, stderr=log.stderr)

    def modify_bin_wrappers(self):
        for path in self.repo_path().joinpath('bin-wrappers').iterdir():
            path.write_bytes(BIN_WRAPPER_TEMPLATE.format(filename=path.name))

    @staticmethod
    def remove_empty_dirs(dir_path):
        for path in dir_path.iterdir():
            if path.is_dir():
                BrowserBuildGit.remove_empty_dirs(path)
        if not list(dir_path.iterdir()):
            logging.debug('Remove %s', dir_path)
            dir_path.rmdir()

    def clean_trash(self):
        shutil.rmtree(str(self.repo_path().joinpath('.git')))
        for ext in ('a', 'c', 'h', 'o', 'o.d'):
            for path in self.repo_path().glob('**/*.{}'.format(ext)):
                logging.debug('Remove %s', path)
                path.unlink()
        self.remove_empty_dirs(self.repo_path())

    def publish_resource(self):
        archive_path = str(self.path('git.tar.gz'))
        with tarfile.open(archive_path, mode='w:gz') as tf:
            tf.add(str(self.repo_path()), arcname='')

        resource = BrowserGitEnvironment(self, 'Git', archive_path)
        if self.Parameters.git_version:
            resource.version = self.Parameters.git_version
        sdk2.ResourceData(resource).ready()

    def on_execute(self):
        self.clone_git()
        self.build_git()
        self.modify_bin_wrappers()
        self.clean_trash()
        self.publish_resource()
