import itertools
import logging
import re
import urlparse

import sandbox.common.types.client as ctc
from sandbox.common.errors import TaskFailure
from sandbox.common.types import notification as ctn
from sandbox.common.types import task as ctt

from sandbox.projects.common import decorators

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox import common
from sandbox import sdk2


class Build(object):

    class DisplayStyle(object):
        SUCCESS = "status_success"
        EXCEPTION = "status_exception"
        DRAFT = "status_draft"
        WAIT = "status_wait_time"
        EXECUTING = "status_executing"
        DELETED = "status_deleted"

    @classmethod
    def scheme(cls):
        raise NotImplementedError

    def __init__(self, type_, status):
        self.type = type_
        self.status = status

    def update(self):
        raise NotImplementedError

    def finished(self):
        raise NotImplementedError

    def url(self):
        raise NotImplementedError

    def pickle(self):
        raise NotImplementedError

    def sandbox_wait_spec(self):
        raise NotImplementedError

    @classmethod
    def unpickle(cls, parameters, data):
        raise NotImplementedError


class SandBoxBuild(Build):

    class Status(object):
        SUCCESS = ['SUCCESS', Build.DisplayStyle.SUCCESS]

        ENQUEUING = ['ENQUEUING', Build.DisplayStyle.WAIT]
        ENQUEUED = ['ENQUEUED', Build.DisplayStyle.WAIT]
        ASSIGNED = ['ASSIGNED', Build.DisplayStyle.WAIT]
        WAIT_TIME = ['WAIT_TIME', Build.DisplayStyle.WAIT]
        WAIT_TASK = ['WAIT_TASK', Build.DisplayStyle.WAIT]
        STOPPING = ['STOPPING', Build.DisplayStyle.WAIT]

        EXECUTING = ['EXECUTING', Build.DisplayStyle.EXECUTING]
        PREPARING = ['PREPARING', Build.DisplayStyle.EXECUTING]
        FINISHING = ['FINISHING', Build.DisplayStyle.EXECUTING]

        FAILURE = ['FAILURE', Build.DisplayStyle.EXCEPTION]
        EXCEPTION = ['EXCEPTION', Build.DisplayStyle.EXCEPTION]
        STOPPED = ['STOPPED', Build.DisplayStyle.EXCEPTION]
        TIMEOUT = ['TIMEOUT', Build.DisplayStyle.EXCEPTION]
        UNKNOWN = ['UNKNOWN', Build.DisplayStyle.EXCEPTION]
        UNKNOWN_FINISHED = ['UNKNOWN_FINISHED', Build.DisplayStyle.EXCEPTION]

        DRAFT = ['DRAFT', Build.DisplayStyle.DRAFT]
        DELETED = ['DELETED', Build.DisplayStyle.DELETED]

    SANDBOX_FINISHED_STATUSES = (
        Status.SUCCESS,
        Status.FAILURE,
        Status.EXCEPTION,
        Status.STOPPED,
        Status.DELETED,
        Status.TIMEOUT
    )

    STATUSES_TO_WAIT_FOR = [
        status for status, style in SANDBOX_FINISHED_STATUSES
    ]

    FINISHED_STATUSES = SANDBOX_FINISHED_STATUSES + (Status.UNKNOWN_FINISHED,)

    def __init__(self, type_, parameters, server, id_,
                 status=Status.UNKNOWN, web_url=None):
        super(SandBoxBuild, self).__init__(type_, status)
        self.server = server
        self.task_id = id_
        self.web_url = web_url

    @classmethod
    def scheme(cls):
        return 'sb'

    def update(self):
        if not self.finished():
            builds = sdk2.Task.find(id=self.task_id).limit(1)
            if builds.count == 1:
                build = builds.first()
                self.type = str(build.type)  # type is object
                self.status = SandBoxBuild.Status.__dict__.get(
                    build.status)
                if self.status == SandBoxBuild.Status.UNKNOWN:
                    self.status = SandBoxBuild.Status.UNKNOWN_FINISHED
                if not self.status:
                    raise TaskFailure('Unknown build status: {}'.format(
                        build.status))
            else:
                raise TaskFailure('Unknown task id: {}'.format(self.task_id))

    def url(self):
        return 'https://{server}/task/{task_id}/view'.format(
            server=self.server, task_id=self.task_id)

    def pickle(self):
        return {'type': self.type, 'status': self.status,
                'server': self.server, 'task_id': self.task_id,
                'web_url': self.web_url}

    @classmethod
    def unpickle(cls, parameters, data):
        return SandBoxBuild(data['type'], parameters, data['server'],
                            data['task_id'], data['status'], data['web_url'])

    def sandbox_wait_spec(self):
        return self.task_id, self.STATUSES_TO_WAIT_FOR

    def finished(self):
        return self.status in self.FINISHED_STATUSES


# BYIN-12318: change this to a more robust method
def get_sandbox_task_id_to_wait(
        build, wait_reason='Waiting for sandbox task...',
        comment_re=re.compile(r'^Created sandbox task sandbox-(\d+)$')):
    if build.data.get('waitReason', None) == wait_reason:
        if build.comment:
            lines = build.comment.splitlines()
            if lines:
                match = comment_re.match(lines[-1])
                if match:
                    return int(match.group(1))


class TeamcityBuild(Build):

    BROWSER_TEAMCITY = 'teamcity.browser.yandex-team.ru'
    COMMON_TEAMCITY = 'teamcity.yandex-team.ru'
    TAXI_TEAMCITY = 'teamcity.taxi.yandex-team.ru'

    class Status(object):
        SUCCESS = ['SUCCESS', Build.DisplayStyle.SUCCESS]

        QUEUED = ['QUEUED', Build.DisplayStyle.WAIT]

        RUNNING = ['RUNNING', Build.DisplayStyle.EXECUTING]

        FAILURE = ['FAILURE', Build.DisplayStyle.EXCEPTION]
        UNKNOWN = ['UNKNOWN', Build.DisplayStyle.EXCEPTION]
        UNKNOWN_FINISHED = ['UNKNOWN_FINISHED', Build.DisplayStyle.EXCEPTION]

    @classmethod
    def scheme(cls):
        return 'tc'

    @classmethod
    def _get_token_source(cls, server, parameters):
        if server == cls.BROWSER_TEAMCITY:
            return parameters.browser_teamcity_vault
        elif server == cls.COMMON_TEAMCITY:
            return parameters.common_teamcity_vault
        elif server == cls.TAXI_TEAMCITY:
            return parameters.taxi_teamcity_vault
        else:
            raise TaskFailure("No token source for TeamCity server "
                              "{}".format(server))

    def __init__(self, type_, parameters, server, id_, sandbox_task_id=None,
                 status=Status.UNKNOWN, web_url=None):
        super(TeamcityBuild, self).__init__(type_, status)
        self.token_source = self._get_token_source(server, parameters)
        self.server = server
        self.build_id = id_
        self.sandbox_task_id = sandbox_task_id
        self.web_url = web_url

    @classmethod
    @common.utils.singleton
    def _teamcity_client(cls, server, token_source):
        import teamcity_client.client
        return teamcity_client.client.TeamcityClient(
            server_url=server, auth=sdk2.Vault.data(token_source))

    def url(self):
        return self.web_url

    @decorators.retries(5, delay=0.5, backoff=24)
    def update(self):
        from requests.exceptions import HTTPError

        if not self.finished():
            client = self._teamcity_client(self.server, self.token_source)
            try:
                build = client.builds[self.build_id]
                self.type = build.build_type.id
                self.web_url = build.web_url
                self.sandbox_task_id = get_sandbox_task_id_to_wait(build)
                if self.sandbox_task_id:
                    logging.info(
                        'The build at %s is waiting for sandbox task %d',
                        self.web_url, self.sandbox_task_id)

                if build.state == 'queued':
                    self.status = TeamcityBuild.Status.QUEUED
                elif build.state == 'running':
                    self.status = TeamcityBuild.Status.RUNNING
                elif build.state == 'finished':
                    if build.status == 'SUCCESS':
                        self.status = TeamcityBuild.Status.SUCCESS
                    elif build.status == 'FAILURE':
                        self.status = TeamcityBuild.Status.FAILURE
                    elif build.status == 'UNKNOWN':
                        self.status = TeamcityBuild.Status.UNKNOWN_FINISHED
                    else:
                        logging.warning(
                            'Unknown finished build status for %s: %s',
                            self.web_url, build.status)
                        self.status = TeamcityBuild.Status.UNKNOWN_FINISHED
                else:
                    logging.warning('Unknown build state for %s: %s',
                                    self.web_url, build.state)
                    self.status = TeamcityBuild.Status.UNKNOWN_FINISHED
            except HTTPError as e:
                if e.response.status_code == 404:
                    logging.error('Build with id %s was not found on %s',
                                  self.build_id, self.server)
                    self.status = TeamcityBuild.Status.UNKNOWN_FINISHED
                else:
                    raise

    def pickle(self):
        return {'type': self.type, 'status': self.status,
                'server': self.server, 'build_id': self.build_id,
                'sandbox_task_id': self.sandbox_task_id,
                'web_url': self.web_url}

    @classmethod
    def unpickle(cls, parameters, data):
        return TeamcityBuild(data['type'], parameters, data['server'],
                             data['build_id'], data['sandbox_task_id'],
                             data['status'], data['web_url'])

    def sandbox_wait_spec(self):
        return (
            (self.sandbox_task_id, SandBoxBuild.STATUSES_TO_WAIT_FOR)
            if self.sandbox_task_id else None
        )

    def finished(self):
        return self.status in (TeamcityBuild.Status.SUCCESS,
                               TeamcityBuild.Status.FAILURE,
                               TeamcityBuild.Status.UNKNOWN_FINISHED)


BUILD_HANDLERS = {handler.scheme(): handler
                  for handler in [TeamcityBuild, SandBoxBuild]}


class BrowserWaitPerfBuilds(sdk2.Task):

    class Requirements(sdk2.Requirements):
        disk_space = 100
        client_tags = ctc.Tag.BROWSER  # because of teamcity access
        cores = 1
        environments = [
            PipEnvironment('teamcity-client', '4.8.2'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        kill_timeout = 60 * 60

        notifications = (
            sdk2.Notification(
                (ctt.Status.FAILURE, ctt.Status.EXCEPTION, ctt.Status.NO_RES,
                 ctt.Status.TIMEOUT),
                ('malets', 'terkira',),
                ctn.Transport.EMAIL
            ),
        )

        builds = sdk2.parameters.String(
            "Builds to wait for (space-separated). Format is:"
            "<type>://<server>/<id>")

        final_wait = sdk2.parameters.Integer(
            "Wait this time (in seconds) after all the builds to wait for "
            "have finished.  May help to smooth out peak load."
        )

        sleep_time = sdk2.parameters.Integer("Sleep time (minutes)", default=5)

        with sdk2.parameters.Group("Credentials") as credentials_group:
            browser_teamcity_vault = sdk2.parameters.String(
                'Vault item with token for browser teamcity',
                default='ROBOT_SPEEDY_BROWSER_TEAMCITY')
            common_teamcity_vault = sdk2.parameters.String(
                'Vault item with token for common teamcity',
                default='ROBOT_SPEEDY_COMMON_TEAMCITY')
            taxi_teamcity_vault = sdk2.parameters.String(
                'Vault item with token for taxi teamcity',
                default='ROBOT_SPEEDY_TAXI_TEAMCITY')

    class Context(sdk2.Context):
        sbuilds = None

    def load_build(self, scheme, data):
        return BUILD_HANDLERS[scheme].unpickle(self.Parameters, data)

    def load(self):
        if self.Context.sbuilds is None:
            self.Context.sbuilds = []
            builds = []
            failed = []
            for spec in self.Parameters.builds.split():
                parsed_spec = urlparse.urlparse(spec)
                handler_cls = BUILD_HANDLERS.get(parsed_spec.scheme)
                if not handler_cls:
                    failed.append(spec)
                else:
                    build = handler_cls(
                        None, self.Parameters, parsed_spec.netloc,
                        parsed_spec.path[1:])
                    self.Context.sbuilds.append(
                        (build.scheme(), build.pickle()))
                    builds.append(build)
            if failed:
                raise TaskFailure("The following builds have unknown handlers:"
                                  "{}".format(" ".join(failed)))
            return builds
        else:
            return [self.load_build(scheme, data)
                    for (scheme, data) in self.Context.sbuilds]

    def save(self, builds):
        self.Context.sbuilds = [(build.scheme(), build.pickle())
                                for build in builds]

    def update(self, builds):
        success = True
        for build in builds:
            try:
                build.update()
            except Exception:
                logging.exception('Failed to get build status for %s',
                                  build.url())
                success = False
        return success

    def on_execute(self):
        builds = self.load()
        if not self.update(builds):
            self.set_info('Failed to update builds status')

        self.save(builds)

        if any(not build.finished() for build in builds):
            specs_to_wait = zip(*filter(
                None, (build.sandbox_wait_spec() for build in builds
                       if not build.finished())
            ))
            if specs_to_wait:
                tasks_to_wait, statuses_to_wait = specs_to_wait
                raise sdk2.WaitTask(tasks_to_wait,
                                    itertools.chain(*statuses_to_wait))
            else:
                raise sdk2.WaitTime(self.Parameters.sleep_time * 60)

        if self.Parameters.final_wait and not self.Context.final_wait:
            self.Context.final_wait = self.Parameters.final_wait
            raise sdk2.WaitTime(self.Parameters.final_wait)

    @sdk2.header()
    def header(self):
        rows = []

        if self.Context.sbuilds:
            for (scheme, data) in self.Context.sbuilds:
                build = self.load_build(scheme, data)
                rows.append({
                    "Status": "<span class='status {}'>{}</span>".format(
                        build.status[1], build.status[0]),
                    "Build type": build.type,
                    "URL": "<a href='{0}'>{0}</a>".format(build.url())
                })

        return [{
            "content": {"": rows},
            "helperName": "",
        }]
