import sandbox.common.types.client as ctc
from sandbox.projects.browser.common import bitbucket
from sandbox.projects.browser.util.BrowserWaitTeamcityBuilds import BrowserWaitTeamcityBuilds

from sandbox.common.types.task import Status
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox import sdk2

TEAMCITY_URL = 'https://teamcity.browser.yandex-team.ru'


class BrowserWaitPrBuilds(sdk2.Task):
    class Requirements(sdk2.Requirements):
        disk_space = 100
        client_tags = ctc.Tag.BROWSER  # because of bitbucket access
        cores = 1
        environments = [
            PipEnvironment('teamcity-client==4.0.0'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        project = sdk2.parameters.String("Bitbucket project", default='stardust')
        repo = sdk2.parameters.String("Bitbucket repo", default='browser')
        pr_id = sdk2.parameters.Integer("Bitbucket pull request id")

        sleep_time = sdk2.parameters.Integer("Sleep time (minutes)", default=15)
        finish_after_first_fail = sdk2.parameters.Bool(
            "Finish task after some teamcity build fails", default=False)

        oauth_vault = sdk2.parameters.String('Vault item with token for teamcity',
                                             default='robot-browser-infra_teamcity_token')

    class Context(sdk2.Context):
        builds = []

    @property
    def teamcity_client(self):
        from teamcity_client import client
        return client.TeamcityClient(
            server_url='teamcity.browser.yandex-team.ru',
            auth=sdk2.Vault.data(self.Parameters.oauth_vault)
        )

    @property
    def bb(self):
        return bitbucket.BitBucket(
            bitbucket.DEFAULT_BITBUCKET_URL,
            'x-oauth-token',
            sdk2.Vault.data(self.Parameters.oauth_vault))

    def description(self, pr):
        return '{}: {}'.format(
            '<a href={}>PR#{}</a>'.format(pr.web_url, pr.id),
            pr.title
        )

    def on_execute(self):
        if not self.Context.builds:
            self.Context.builds = [
                build['id'] for build in self.bb.get_pull_request_builds(
                    self.Parameters.project, self.Parameters.repo, self.Parameters.pr_id)
                if build['id']
            ]
            pr = self.bb.get_pr(self.Parameters.project, self.Parameters.repo, self.Parameters.pr_id)
            self.Parameters.description = self.description(pr)
            raise sdk2.WaitTask(
                BrowserWaitTeamcityBuilds(
                    self,
                    description='Wait builds for {}'.format(self.description(pr)),
                    mode='WAIT_GIVEN',
                    builds=' '.join(str(build_id) for build_id in self.Context.builds),
                    oauth_vault=self.Parameters.oauth_vault,
                    sleep_time=self.Parameters.sleep_time,
                    finish_after_first_fail=self.Parameters.finish_after_first_fail,
                ).enqueue(),
                list(Status.Group.FINISH + Status.Group.BREAK),
                True,
            )
        else:
            builds = [self.teamcity_client.builds[build_id] for build_id in self.Context.builds]
            finished_builds_numb = len([build for build in builds if build.state == 'finished'])
            self.set_info('{} of {} builds finished'.format(finished_builds_numb, len(builds)))
