from collections import OrderedDict
import jinja2
import logging
import os
import socket
import time

import requests

import sandbox.common.types.client as ctc
from sandbox.common.utils import get_task_link

from sandbox.projects.common import decorators
from sandbox.projects.browser.common.teamcity import run_builds_with_deps
from sandbox.projects.browser.util.BrowserStopTeamcityBuilds import BrowserStopTeamcityBuilds

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox import common
from sandbox import sdk2


class BrowserWaitTeamcityBuilds(sdk2.Task):
    class Requirements(sdk2.Requirements):
        disk_space = 100
        client_tags = ctc.Tag.BROWSER  # because of teamcity access
        cores = 1
        environments = [
            PipEnvironment('teamcity-client==4.8.2'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.String("Mode") as mode:
            mode.values.RUN_NEW = mode.Value("Run new builds", default=True)
            mode.values.WAIT_GIVEN = mode.Value("Wait for given builds")

        with mode.value['RUN_NEW']:
            branch = sdk2.parameters.String("Branch", default="master")
            build_counts = sdk2.parameters.Dict('Builds to trigger')
            put_to_top = sdk2.parameters.Bool("Put builds to the queue top", default=False)
            dependent_builds_count = sdk2.parameters.Integer(
                'Limit for snapshot dependencies count', default=0)

        with mode.value['WAIT_GIVEN']:
            builds = sdk2.parameters.String("Builds to wait for (space-separated)")

        sleep_time = sdk2.parameters.Integer("Sleep time (minutes)", default=15)
        finish_after_first_fail = sdk2.parameters.Bool(
            "Finish task after some teamcity build fails", default=False)
        kill_on_cancel = sdk2.parameters.Bool(
            "Kill builds if this task is canceled", default=False)

        with sdk2.parameters.Group("Timeout") as timeout_group:
            with sdk2.parameters.String("Mode") as timeout_mode:
                timeout_mode.values.NONE = timeout_mode.Value("No timeout", default=True)
                timeout_mode.values.FORGET = timeout_mode.Value("Do not wait for builds to finish after timeout")
                timeout_mode.values.CANCEL = timeout_mode.Value("Cancel unfinished builds on timeout")

            with timeout_mode.value['FORGET'], timeout_mode.value['CANCEL']:
                timeout = sdk2.parameters.Integer("Timeout (minutes)", default=None)

        with sdk2.parameters.Group("Credentials") as credentials_group:
            robot_login = sdk2.parameters.String("Login for teamcity", default="robot-browser-infra")
            robot_password_vault = sdk2.parameters.String("Vault item with password for teamcity",
                                                          default="robot-browser-infra_password")
            oauth_vault = sdk2.parameters.String('Vault item with token for teamcity',
                                                 default='robot-browser-infra_teamcity_token')

    class Context(sdk2.Context):
        finished_builds = []
        running_builds = None
        build_type_stats = {}
        build_status_texts = {}
        start_timestamp = None

    @property
    @common.utils.singleton
    def teamcity_client(self):
        from teamcity_client import client
        return client.TeamcityClient(
            server_url='teamcity.browser.yandex-team.ru',
            auth=(
                sdk2.Vault.data(self.Parameters.oauth_vault) if self.Parameters.oauth_vault
                else (self.Parameters.robot_login, sdk2.Vault.data(self.Parameters.robot_password_vault))
            )
        )

    @decorators.retries(5, delay=0.5, backoff=24, exceptions=(requests.RequestException, socket.error))
    def gather_task_info(self):
        return {
            build_id: self.teamcity_client.builds[build_id]
            for build_id in self.Context.running_builds
        }

    def on_execute(self):
        if self.Context.running_builds is None:
            if self.Parameters.mode == 'RUN_NEW':
                # run builds
                all_builds = []
                for build_type, count in self.Parameters.build_counts.iteritems():
                    count = int(count)  # Dict param type has strings in values
                    builds, _ = run_builds_with_deps(
                        self.teamcity_client,
                        branch=self.Parameters.branch,
                        build_type=build_type,
                        count=count,
                        deps_count=self.Parameters.dependent_builds_count or count,
                        comment='{}. Triggered by {} from {}'.format(
                            self.Parameters.description, self.author, get_task_link(self.id),
                        ),
                        put_to_top=self.Parameters.put_to_top,
                    )
                    all_builds.extend(builds)

                self.Context.running_builds = [b.id for b in all_builds]
            else:
                self.Context.running_builds = map(int, self.Parameters.builds.split())

            builds_info = self.gather_task_info()
            build_types = set(build.data.buildTypeId for build in builds_info.itervalues())

            self.Context.build_type_stats = {
                build_type: {
                    'RUNNING': [],
                    'SUCCESS': [],
                    'FAILURE': [],
                    'UNKNOWN': [],
                } for build_type in build_types
            }

            for build_id, build in builds_info.iteritems():
                self.Context.build_type_stats[build.data.buildTypeId]['RUNNING'].append(build_id)
                self.Context.build_status_texts[build_id] = getattr(build.data, 'statusText', '')

        elif self.Context.running_builds:
            # poll builds
            try:
                running_builds_info = self.gather_task_info()
            except:
                logging.exception('Failed to get running builds info')
                self.set_info('Failed to get running builds info')
            else:
                new_finished_builds_ids = [
                    build_id for build_id, build_object in running_builds_info.iteritems()
                    if build_object.state == 'finished'
                ]
                self.Context.finished_builds.extend(new_finished_builds_ids)
                self.Context.running_builds = list(
                    set(self.Context.running_builds) - set(self.Context.finished_builds)
                )
                for build_id in new_finished_builds_ids:
                    build = running_builds_info[build_id]
                    self.Context.build_type_stats[build.data.buildTypeId]['RUNNING'].remove(build_id)
                    self.Context.build_type_stats[build.data.buildTypeId][build.status].append(build_id)
                    self.Context.build_status_texts[build_id] = getattr(build.data, 'statusText', '')
                if self.Parameters.finish_after_first_fail:
                    for build_type, statuses in self.Context.build_type_stats.iteritems():
                        for status, builds in statuses.iteritems():
                            if status == 'FAILURE' and builds:
                                logging.info('One of builds failed, stopping task')
                                return

        if self.Context.start_timestamp is None:
            self.Context.start_timestamp = int(time.time())

        self.set_info("{} running, {} done".format(
            len(self.Context.running_builds), len(self.Context.finished_builds)
        ))

        if self.Context.running_builds:
            wait_time = self.Parameters.sleep_time * 60

            if self.Parameters.timeout_mode != 'NONE':
                time_left = (
                    (self.Context.start_timestamp + self.Parameters.timeout * 60) - int(time.time())
                )

                if time_left <= 0:
                    if self.Parameters.timeout_mode == 'FORGET':
                        self.set_info('Timeout is reached, stop waiting for builds')
                    elif self.Parameters.timeout_mode == 'CANCEL':
                        self.set_info('Timeout is reached, killing builds')
                        BrowserStopTeamcityBuilds(
                            self,
                            description='Stop builds on timeout for task {}'.format(self.id),
                            owner=self.Parameters.owner,
                            teamcity_build_ids=self.Context.running_builds,
                            cancel_comment='Task sandbox-{} timed out'.format(self.id),
                            oauth_vault=self.Parameters.oauth_vault,
                        ).enqueue()

                    return
                else:
                    wait_time = min(wait_time, time_left)

            raise sdk2.WaitTime(wait_time)

    def on_break(self, prev_status, status):
        if self.Parameters.kill_on_cancel and self.Context.running_builds:
            BrowserStopTeamcityBuilds(
                self,
                description='Stop builds for task {}'.format(self.id),
                owner=self.Parameters.owner,
                teamcity_build_ids=self.Context.running_builds,
                cancel_comment='Task sandbox-{} was stopped'.format(self.id),
                oauth_vault=self.Parameters.oauth_vault,
            ).enqueue()

    STATUSES_CLASSES = OrderedDict([
        ("RUNNING", "status_executing"),
        ("SUCCESS", "status_success"),
        ("FAILURE", "status_exception"),
        ("UNKNOWN", "status_draft")
    ])

    @sdk2.header()
    def header(self):
        template_path = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path), extensions=['jinja2.ext.do'])
        return env.get_template('header.html').render({
            'build_type_stats_items': self.Context.build_type_stats.iteritems(),
            'STATUSES_CLASSES': self.STATUSES_CLASSES,
            'branch': self.Parameters.branch,
        })

    @sdk2.footer()
    def footer(self):
        build_descriptions = OrderedDict()
        for build_type, stats in self.Context.build_type_stats.iteritems():
            for status, builds in stats.iteritems():
                for build_id in builds:
                    status_text = ''
                    # new fields in Context break old builds display
                    if hasattr(self.Context, 'build_status_texts'):
                        status_text = self.Context.build_status_texts.get(str(build_id), '')

                    build_descriptions[build_id] = {
                        'status': status,
                        'status_class': self.STATUSES_CLASSES[status],
                        'status_text': status_text,
                        'build_type': build_type,
                    }

        template_path = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path), extensions=['jinja2.ext.do'])
        return env.get_template('footer.html').render({
            'build_descriptions': build_descriptions,
        })
