import datetime
import itertools
import logging
import os
import requests
import socket

import pytz

from sandbox.common import utils
from sandbox.common.errors import TaskFailure, TemporaryError
from sandbox.common.types.client import Tag
from sandbox.common.types.task import Status

from sandbox.projects.browser.common.bitbucket import DEFAULT_BITBUCKET_URL
from sandbox.projects.browser.util.BrowserStopTeamcityBuilds import BrowserStopTeamcityBuilds
from sandbox.projects.browser.util.BrowserWaitTeamcityBuilds import BrowserWaitTeamcityBuilds
from sandbox.projects.browser.util.configurable_trigger import (
    load_config_if_valid, launch_builds_by_config,
)
from sandbox.projects.common import decorators
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox import sdk2

TEAMCITY_URL = 'teamcity.browser.yandex-team.ru'
CONFIGS_DIR = os.path.join(
    os.path.dirname(os.path.dirname(__file__)),
    'configs',
)


class BrowserConfigurableTeamcityTrigger(sdk2.Task):
    class Requirements(sdk2.Requirements):
        disk_space = 5 * 1024
        client_tags = Tag.BROWSER & Tag.Group.LINUX
        cores = 1
        environments = [
            PipEnvironment('PyYAML', version='3.11'),
            PipEnvironment('jsonschema', version='2.5.1'),
            PipEnvironment('teamcity-client==4.7.0'),
            PipEnvironment('bitbucket-server-client==2.18.0'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        branch = sdk2.parameters.String('Branch to run builds on', default='master', required=True)
        config_path = sdk2.parameters.String('Path to config file in the config repo', required=True)
        sleep_time = sdk2.parameters.Integer('Sleep time (minutes)', default=15)
        teamcity_build_id = sdk2.parameters.String('ID of parent teamcity build', required=True)

        with sdk2.parameters.Group("Timeout") as timeout_group:
            with sdk2.parameters.String("Mode") as timeout_mode:
                timeout_mode.values.NONE = timeout_mode.Value(
                    "No timeout", default=True)
                timeout_mode.values.ABSOLUTE = timeout_mode.Value(
                    "Absolute (time to stop builds at, HH:MM, 24-hour, MSK)")
                timeout_mode.values.RELATIVE = timeout_mode.Value(
                    "Relative (minutes)")

            with timeout_mode.value['ABSOLUTE'], timeout_mode.value['RELATIVE']:
                timeout = sdk2.parameters.String('Timeout', default=None)

        with sdk2.parameters.Group('Credentials') as credentials_group:
            oauth_vault = sdk2.parameters.String('Vault item with token for teamcity',
                                                 default='robot-browser-infra_teamcity_token')

    class Context(sdk2.Context):
        launched_builds = {}

    @utils.singleton_property
    def teamcity_client(self):
        import teamcity_client.client
        return teamcity_client.client.TeamcityClient(
            server_url=TEAMCITY_URL,
            auth=(
                sdk2.Vault.data(self.Parameters.oauth_vault)
            )
        )

    @utils.singleton_property
    def bitbucket_client(self):
        import bitbucket
        return bitbucket.BitBucket(
            url=DEFAULT_BITBUCKET_URL,
            token=sdk2.Vault.data(self.Parameters.oauth_vault)
        )

    @decorators.retries(5, delay=0.5, backoff=2, exceptions=(requests.RequestException, socket.error))
    def get_change_id(self):
        revisions = self.teamcity_client.builds[self.Parameters.teamcity_build_id].revisions
        if not revisions:
            return None
        else:
            version = revisions[0].version
            vcs_root_id = revisions[0].get('vcs-root-instance', {}).get('vcs-root-id', '')

        change = self.teamcity_client.Change(version=version, vcsRoot=vcs_root_id)
        logging.info('Collected change %s', change)
        if not change:
            return None
        else:
            return change

    def build_comment(self, config_name):
        return 'Launched by trigger sandbox-{}, config {}'.format(self.id, config_name)

    def on_execute(self):
        timeout = None

        if self.Parameters.timeout_mode == 'ABSOLUTE':
            hh, mm = map(int, self.Parameters.timeout.split(':'))
            assert 0 <= hh < 24
            assert 0 <= mm < 60
            now = datetime.datetime.now(pytz.timezone('Europe/Moscow'))
            # absolute timeout is either today or tomorrow
            # let's calculate timeout if it is today and check
            timeout_today = now.replace(second=0, minute=mm, hour=hh)
            if timeout_today > now:
                # timeout is today
                timeout_time = timeout_today
            else:  # timeout_today <= now
                # timeout is tomorrow
                timeout_time = timeout_today + datetime.timedelta(days=1)

            timeout = (timeout_time - now).total_seconds() / 60
        elif self.Parameters.timeout_mode == 'RELATIVE':
            timeout = int(self.Parameters.timeout)

        with self.memoize_stage.trigger_builds:
            full_config_path = os.path.join(CONFIGS_DIR, self.Parameters.config_path)
            config_dict = load_config_if_valid(full_config_path)

            # we'll launch builds on the same changeset as trigger's
            change = self.get_change_id()
            if change is None:
                raise TemporaryError('Change not available yet, waiting')

            # remove refs/heads/ from branch name
            branch = self.Parameters.branch
            if branch.startswith('refs/heads/'):
                branch = branch[11:]

            for config_name, config in config_dict.iteritems():
                if config_name not in self.Context.launched_builds:
                    builds = launch_builds_by_config(
                        bb=self.bitbucket_client,
                        tc=self.teamcity_client,
                        config_path=self.Parameters.config_path,
                        config_name=config_name,
                        config=config,
                        branch=branch,
                        change=change,
                        comment=self.build_comment(config_name),
                    )

                    if not builds:
                        self.set_info(
                            'No builds were triggered on branch {} for config {}'.format(
                                branch, config_name
                            )
                        )
                    else:
                        self.set_info(
                            'Triggered {} builds on branch {} for config {}'.format(
                                len(builds), branch, config_name
                            )
                        )
                    self.Context.launched_builds[config_name] = [b.id for b in builds]

            all_builds = [
                build_id
                for config_builds in self.Context.launched_builds.itervalues()
                for build_id in config_builds
            ]

            if all_builds:
                raise sdk2.WaitTask(
                    BrowserWaitTeamcityBuilds(
                        self,
                        mode='WAIT_GIVEN',
                        builds=' '.join(str(build_id) for build_id in all_builds),
                        oauth_vault=self.Parameters.oauth_vault,
                        sleep_time=self.Parameters.sleep_time,
                        timeout_mode='CANCEL' if timeout is not None else 'NONE',
                        timeout=timeout,
                    ).enqueue(),
                    list(Status.Group.FINISH + Status.Group.BREAK),
                    True,
                )
        if not all(task.status == Status.SUCCESS for task in self.find()):
            raise TaskFailure('Some of launched builds failed')

    def on_break(self, prev_status, status):
        if self.Context.launched_builds:
            BrowserStopTeamcityBuilds(
                self,
                description='Stop builds for task {}'.format(self.id),
                owner=self.Parameters.owner,
                teamcity_build_ids=list(itertools.chain.from_iterable(self.Context.launched_builds.values())),
                cancel_comment='Task sandbox-{} was stopped'.format(self.id),
            ).enqueue()

    @sdk2.header()
    def header(self):
        wait_task = self.find(BrowserWaitTeamcityBuilds).first()
        if wait_task:
            return wait_task.header()
        else:
            return []

    @sdk2.footer()
    def footer(self):
        wait_task = self.find(BrowserWaitTeamcityBuilds).first()
        if wait_task:
            return wait_task.footer()
        else:
            return []
