import logging
import os

from sandbox.common.errors import TaskFailure
from sandbox.common.types.client import Tag

from sandbox.projects.browser.common.git import repositories
from sandbox.projects.browser.util.configurable_trigger import load_config_if_valid
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox import sdk2


class BrowserValidateTriggerConfig(sdk2.Task):
    class Requirements(sdk2.Requirements):
        disk_space = 5 * 1024
        client_tags = Tag.BROWSER & Tag.Group.LINUX
        cores = 1
        environments = [
            PipEnvironment('PyYAML', version='3.11'),
            PipEnvironment('jsonschema', version='2.5.1'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        branch = sdk2.parameters.String('Branch to checkout and validate configs', required=True)
        commit = sdk2.parameters.String('Commit to checkout and validate configs')

    class Context(sdk2.Context):
        pass

    def on_execute(self):
        repo_path = self.path('build-launcher-configs')
        vcs_root = repositories.Stardust.build_launcher_configs(filter_branches=False)
        vcs_root.update_cache_repo()
        vcs_root.clone(str(repo_path), branch=self.Parameters.branch, commit=self.Parameters.commit)
        failed_configs = []
        for dir_name, _, files in os.walk(str(repo_path)):
            if '.git' in dir_name:
                continue
            for file_name in files:
                if not file_name.endswith('.yaml'):
                    continue
                full_config_path = os.path.join(dir_name, file_name)

                try:
                    load_config_if_valid(full_config_path)
                except Exception as e:
                    logging.exception('Unable to load or validate config %s', full_config_path)
                    self.set_info('Error for config {}: {}'.format(full_config_path, e.message))
                    failed_configs.append(full_config_path)

        if failed_configs:
            raise TaskFailure('Unable to load or validate configs: {}'.format('\n'.join(failed_configs)))
