import datetime
import itertools
import fnmatch
import functools
import logging
import operator
import re
import socket
import time
import yaml

import requests

from sandbox.projects.common import decorators

from sandbox.projects.browser.common.teamcity import run_builds_with_deps
from sandbox.projects.browser.util.configurable_trigger.config_schema import SCHEMA


RELEASES_INFO_URL = (
    'https://browser.yandex-team.ru/rest/projects/desktop/releases/'
    '?include_milestones=true&show_only_active=true')

MILESTONE_OPERATORS = {
    'milestone_gt': operator.gt,
    'milestone_gte': operator.ge,
    'milestone_lt': operator.lt,
    'milestone_lte': operator.le,
}
PLATFORM_RULES = 'platforms'
COUNT_RULE = 'count'
ORDER_RULE = 'order'

# if many files were changed between commits we'd launch builds w/o checking file filters
CHANGE_LOOKUP_LIMIT = 250


@functools.total_ordering
class Release(object):
    def __init__(self, release, current_timestamp):
        self.branch = release['branch']
        self.milestones = release['milestones']
        self.current_timestamp = current_timestamp

    def __lt__(self, milestone):
        for m in self.milestones:
            if m['kind'] == milestone:
                return self.current_timestamp < m['date']
        return False

    def __eq__(self, milestone):
        for i, m in enumerate(self.milestones):
            if m['kind'] == milestone:
                try:
                    return (
                        self.current_timestamp >= m['date'] and
                        self.current_timestamp < self.milestones[i + 1]['date']
                    )
                except IndexError:
                    return True
        return False


def validate_branch_rules(config_dict):
    for config in config_dict.itervalues():
        branch_rules = config['branches']
        for rule in branch_rules:
            try:
                assert rule.startswith(('+:', '-:'))
                exact_regexp = '^{}$'.format(rule[2:])
                re.compile(exact_regexp)
            except Exception as e:
                raise Exception(
                    'Invalid branch rule: {}\n{}'.format(rule, e.message))


def validate_file_rules(config_dict):
    for config in config_dict.itervalues():
        file_globs = config.get('files', [])
        for file_glob in file_globs:
            if not file_glob.startswith(('+:', '-:')):
                raise Exception(
                    'Invalid file rule: {}\nMust start with +: or -:'.format(file_glob))


@decorators.retries(5, delay=0.5, backoff=2, exceptions=(requests.RequestException, socket.error))
def get_releases_info():
    try:
        r = requests.get(RELEASES_INFO_URL)
        return r.json()['items']
    except Exception as e:
        raise Exception(
            'Problems with getting releases info\n{}'.format(e.message))


def release_fits(release, conditions, current_timestamp):
    r = Release(release, current_timestamp)
    return all(MILESTONE_OPERATORS[op](r, name) for op, name in conditions.iteritems()
               if op in MILESTONE_OPERATORS.iterkeys())


def platform_fits(release, platform_rules):
    if platform_rules is None:
        return True
    return bool(set(release.get('platforms', [])) & set(platform_rules))


def get_current_timestamp():
    return int(time.mktime(datetime.datetime.today().timetuple()))


def unfold_branch_rules_stamp(config_dict, releases_info, timestamp):
    """
    Prepare branch rules list according to filter rules (milestones, platforms)
    """
    for config in config_dict.itervalues():
        new_branch_rules = []
        for rule in config['branches']:
            if isinstance(rule, dict):
                try:
                    # Rule operator can be either + or -
                    rule_operator, conditions = next(rule.iteritems())
                    platform_rules = conditions.get(PLATFORM_RULES)
                    suitable_releases = []
                    for release in releases_info:
                        if (release_fits(release, conditions, timestamp) and
                                platform_fits(release, platform_rules)):
                            suitable_releases.append(release)
                    suitable_releases.sort(
                        reverse=conditions.get(ORDER_RULE, 'asc') == 'desc',
                        key=lambda x: map(int, x['version'].split('.'))
                    )
                    count = conditions.get(COUNT_RULE)
                    if count:
                        suitable_releases = suitable_releases[:count]
                    for release in suitable_releases:
                        assert release['real_branch'], '{} has no "real_branch" field'.format(release)
                        new_branch_rules.append(
                            '{}:{}'.format(rule_operator, release['real_branch'])
                        )
                except Exception as e:
                    raise Exception(
                        'Invalid branch rule: {}\n{}'.format(
                            rule, e.message))
            else:
                new_branch_rules.append(rule)
        config['branches'] = new_branch_rules


def unfold_filter_rules(config_dict):
    releases_info = get_releases_info()
    current_timestamp = get_current_timestamp()
    unfold_branch_rules_stamp(config_dict, releases_info, current_timestamp)


def read_config(config_path):
    with open(config_path) as f:
        try:
            return yaml.safe_load(f)
        except Exception:
            logging.exception('Cannot load config from %s', config_path)
            raise


def validate_config(config):
    import jsonschema
    try:
        jsonschema.validate(config, SCHEMA)
    except Exception:
        logging.warning('Invalid config: %s', config)
        raise

    # TODO validate_branch_rules(config)


def load_config_if_valid(config_path):
    config = read_config(config_path)
    validate_config(config)
    unfold_filter_rules(config)
    validate_branch_rules(config)
    validate_file_rules(config)
    return config


def get_relevant_branches(branches, branch_rules):
    return {branch for branch in branches if is_relevant_branch(branch, branch_rules)}


def is_relevant_branch(branch, branch_rules):
    logging.debug('Matching branch %s against rules: \n%s', branch, '\n'.join(branch_rules))
    for rule in reversed(branch_rules):  # we use the last matching branch rule (like in BitBucket)
        flag, regexp = rule.split(':', 1)
        exact_regexp = '^{}$'.format(regexp)
        if re.match(exact_regexp, branch):
            if flag == '+':
                return True
            if flag == '-':
                return False
            break
    return False


def compare_globs_to_changes(last_processed_revision, trigger_revision, repo, file_globs):
    """
    Support for file glob-like filters in trigger configs
    Globs must use fnmatch notation (https://docs.python.org/2.7/library/fnmatch.html)

    This methods acquires a set of changes file names
        and then calls filter_files_against_globs to match them with globs

    Changes are computed between revisions of the last successful/queued build for a build_type
        and trigger's own teamcity build (trigger_revision)
    """
    if not file_globs:
        raise ValueError('Something is horribly wrong. File globs can not be empty.')

    # Rest API quote:
    #   Gets the file changes available in the from commit but not in the to commit.
    # So to get info about changes in current revision, pass it as `from_commit`.
    change_iterator = repo.compare_changes(
        to_commit=last_processed_revision, from_commit=trigger_revision,
    )
    changes = list(itertools.islice(change_iterator, CHANGE_LOOKUP_LIMIT))

    if len(changes) == CHANGE_LOOKUP_LIMIT:
        logging.info(
            'Too many changes between revisions %s and %s; launching the build anyway',
            last_processed_revision, trigger_revision,
        )
        return True

    changed_files = {change.path.to_string for change in changes}
    if changed_files:
        return filter_files_against_globs(changed_files, file_globs)
    else:
        logging.info(
            'No changed files were found between revisions %s and %s',
            last_processed_revision, trigger_revision
        )
        return False


def filter_files_against_globs(changed_filenames, file_globs):
    if not file_globs or not changed_filenames:
        raise ValueError('Something is horribly wrong. File globs and changed files can not be empty.')

    logging.debug('All changes between commits: %s', changed_filenames)
    files = set()
    for rule in file_globs:
        flag, glob = rule.split(':', 1)
        logging.debug('Matching rule "%s"', rule)
        if flag == '+':
            files.update(fnmatch.filter(changed_filenames, glob))
        else:
            files.difference_update(fnmatch.filter(changed_filenames, glob))

    logging.debug('Result file set: %s', files)
    return bool(files)


def launch_builds_by_config(bb, tc, config_path, config_name, config, branch, change, comment):
    if not is_relevant_branch(branch, config['branches']):
        logging.info('Branch %s does not match rules in the config %s',
                     branch, config_name)
        return []

    build_types = config['build_types']
    count = config.get('count', 1)
    deps_count = config.get('deps_count', 0)
    parameters = config.get('parameters', {})
    file_globs = config.get('files', [])
    tags = config.get('tags', []) + [
        'trigger:{}'.format(config_path),
        'trigger:{}:{}'.format(config_path, config_name),
    ]
    run_with_no_changes = config.get('run_with_no_changes', True)

    logging.info(
        'Will launch %d tasks of types %s on branch %s for config %s',
        count, build_types, branch, config_name,
    )

    launched_builds = []
    def launch_builds():
        builds, deps = run_builds_with_deps(
            tc,
            branch=branch,
            build_type=build_type,
            count=count,
            deps_count=deps_count,
            tags=tags,
            add_deps_tags=True,
            comment=comment,
            change=change,
            parameters=parameters,
        )
        launched_builds.extend(builds)
        launched_builds.extend(deps)

    for build_type in build_types:
        latest_builds = sorted(tc.Builds(
            branch=branch,
            buildType=build_type,
            state='any',
            canceled=False,
            failedToStart=False,
            personal=False,
            history=False,
        ), reverse=True, key=lambda build: build.id)
        if not latest_builds:  # this is the first build on this branch
            logging.info(
                'No recent %s builds on branch %s were found, cannot compare changes, launch for sure',
                build_type, branch
            )
            launch_builds()
            continue

        last_processed_revision = latest_builds[0].revisions[0]['version']
        logging.info('Latest revision for %s: %s', build_type, last_processed_revision)
        if last_processed_revision == change.version and not run_with_no_changes:
            logging.info(
                'Revisions are equal; no need to launch builds; %s == %s',
                last_processed_revision, change.version
            )
            continue

        vcs_root_url = next(
            prop['value']
            for prop in tc.rest_api['vcs-root-instances'].locator(
                id=latest_builds[0].revisions[0]['vcs-root-instance'].id
            ).get()['properties']['property']
            if prop['name'] == 'url'
        )
        if vcs_root_url:
            logging.info('VCS Root URL: %s', vcs_root_url)

            project, repo = re.search('scm/(.+)/(.+).git', vcs_root_url).groups()
            repo = bb.projects[project].repos[repo]
            logging.info('Will use repo %s in the project %s', repo, project)

            globs_fit = not file_globs or compare_globs_to_changes(
                last_processed_revision=last_processed_revision,
                trigger_revision=change.version, repo=repo,
                file_globs=file_globs,
            )
        else:
            globs_fit = True

        if globs_fit:
            launch_builds()

    return launched_builds
