from collections import defaultdict
import os
import pytest
import json

from sandbox.projects.browser.util.configurable_trigger import (
    is_relevant_branch, read_config, validate_config,
    validate_branch_rules, validate_file_rules, unfold_branch_rules_stamp
)


def configs():
    import yatest.common

    configs_dir = os.path.join(
        os.path.dirname(os.path.abspath(yatest.common.test_source_path())),
        'configs',
    )

    for root, _, files in os.walk(configs_dir):
        for filename in files:
            if os.path.splitext(filename)[-1] == '.yaml':
                config = os.path.join(root, filename)

                yield pytest.param(config, id=os.path.relpath(config, configs_dir))


def releases_info_params():
    import yatest.common

    releases_info_dir = os.path.join(
        os.path.dirname(os.path.abspath(yatest.common.test_source_path())),
        'goldentests',
        'releases_info',
    )

    for filename in os.listdir(releases_info_dir):
        timestamp = filename.split('.')[0]
        if not timestamp.isdigit():
            raise ValueError("A name of releases info file must consist of a <timestamp>.json: {}/{}".format(releases_info_dir, filename))

        releases_info_path = os.path.join(releases_info_dir, filename)

        yield pytest.param(
            releases_info_path, int(timestamp),
            id=os.path.relpath(releases_info_path, releases_info_dir)
        )


# FIXME do not test all configs on all branches?
@pytest.mark.parametrize('branch_name', [
    'master',
    'master-mobile',
    'master-next',
    'master-zen-android',
    'master-zen-ios',
    'merge-82.0.4068/0',
    'master-20.7/pre',
    'master-20.3.0/rc',
    'master-20.3.1/rc',
    'master-20.3.1/android/rc',
    'master-20.3.1/ios/rc',
    'master-20.3/android/pre',
    'master-20.3/ios/pre',
    'wp/BROWSER-9999/0',
    'wp-20.3.0/BROWSER-9999/1',
    'wp-mobile/BROWSER-9999/2',
])
@pytest.mark.parametrize('config_path', configs())
@pytest.mark.parametrize(('releases_info_path', 'timestamp'), releases_info_params())
def test_config(branch_name, config_path, releases_info_path, timestamp):
    build_groups = defaultdict(list)

    def find_group(build_type, parameters):
        result = next(
            (g for g in build_groups[build_type] if g.get('parameters', {}) == parameters),
            None,
        )
        if result is None:
            result = {
                'count': 0,
                'deps_count': 0,
                'tags': [],
            }
            if parameters:
                result['parameters'] = parameters
            build_groups[build_type].append(result)

        return result

    with open(releases_info_path) as f:
        releases_info = json.load(f)

    config = read_config(config_path)
    validate_config(config)
    unfold_branch_rules_stamp(config, releases_info, timestamp)
    validate_branch_rules(config)
    validate_file_rules(config)

    for section_name, section in config.iteritems():
        if is_relevant_branch(branch_name, section['branches']):
            for build_type in section['build_types']:
                group = find_group(build_type, section.get('parameters', {}))
                group['count'] += section.get('count', 1)
                group['deps_count'] += section.get('deps_count', 0)
                group['tags'] += section.get('tags', [])

    return dict(build_groups)
