import logging
import os
import shutil

import sandbox.common.types.task as ctt
from sandbox import common
from sandbox import sdk2
from sandbox.projects.browser_mobile.abro.BrowserAndroidTaskBase import BrowserAndroidTaskBase
from sandbox.projects.common.teamcity import TeamcityArtifacts, TeamcityServiceMessagesLog
from sandbox.sdk2.path import Path


def copy_into_dir(src, dst):
    """
    Copy file or directory into target dir.
    Destination dir must exist!
    :param src: path to file or directory.
    :param dst: path to target directory.
    :type src: str or sandbox.sdk2.path.Path
    :type dst: str or sandbox.sdk2.path.Path
    """
    src = str(src)
    dst = str(dst)
    logging.debug('Copy: "%s" into "%s"', src, dst)
    if os.path.isdir(src):
        dst = os.path.join(dst, os.path.basename(src))
        shutil.copytree(src, dst)
    else:
        dst_path = os.path.join(dst, os.path.basename(src))
        if os.path.exists(dst_path):
            logging.debug('Skip copying, destination file is already exist: "%s" into "%s"', src, dst)
        else:
            shutil.copy2(src, dst)


def merge_copy(src, dst):
    """
    Merge directory contents or copy a file into target dir.
    Destination dir must exist!
    :param src: path to file or directory.
    :param dst: path to target directory.
    :type src: str or sandbox.sdk2.path.Path
    :type dst: str or sandbox.sdk2.path.Path
    """
    src = Path(src)
    dst = Path(dst)
    if src.is_dir():
        for src_path in src.glob('**/*'):
            dst_path = dst.joinpath(src_path.relative_to(src))
            if src_path.is_dir():
                if not dst_path.exists():
                    dst_path.mkdir(parents=True)
            else:
                copy_into_dir(src_path, dst_path.parent)
    else:
        copy_into_dir(src, dst)


class OutputResource(sdk2.Resource):
    """ Task output. """
    pass


class GradleDaemonResource(sdk2.Resource):
    """ Contents of '~/.gradle/daemon'. """
    pass


class BrowserAndroidBuildTask(BrowserAndroidTaskBase):
    """ Browser Android build task. """

    class Context(sdk2.Task.Context):
        child_tasks = []

    class Parameters(BrowserAndroidTaskBase.Parameters):
        # custom parameters
        branch = sdk2.parameters.String(
            'Branch reference to build on (e.g. refs/heads/wp/ABRO-12345/0)',
            default='refs/heads/master',
            required=True)
        commit = sdk2.parameters.String(
            'Checkout specific commit to build.')
        yaml_path = sdk2.parameters.String(
            'Path of YAML build configuration.',
            default='build/configs/pr-arm-release.yaml',
            required=True)
        yaml_content = sdk2.parameters.String('Content of YAML build configuration to rewrite.', multiline=True)
        dogfood_branch_name = sdk2.parameters.String('Dogfood branch.', default='master')
        build_defines = sdk2.parameters.List('List of defines in key=value format.', sdk2.parameters.String)
        phone_clids_xml = sdk2.parameters.String('Contents of XML file with clids for phone.')
        tablet_clids_xml = sdk2.parameters.String('Contents of XML file with clids for tablet.')
        single_target = sdk2.parameters.String('Build single target.')
        parallel_targets = sdk2.parameters.Bool('Run each target in parallel task.', default=False, required=True)
        teamcity_build_id = sdk2.parameters.String('Id of teamcity build that triggered this task', required=True)

    def wait_child_tasks_done(self):
        # Check if all child tasks are really done doing something.
        expected_statuses = (ctt.Status.Group.FINISH, ctt.Status.Group.BREAK)
        child_tasks = map(lambda task_id: self.find(id=task_id).first(), self.Context.child_tasks)
        for child in child_tasks:
            if child.status not in expected_statuses:
                logging.info('One or more child tasks still working, returning to WAIT state.')
                raise sdk2.WaitTask(child_tasks, expected_statuses)

    def raise_error_if_child_task_failed(self):
        # Check if all child tasks are in SUCCESS state.
        for child_id in self.Context.child_tasks:
            child = self.find(id=child_id).first()
            logging.info('Child task: {} status {}'.format(child, child.status))
            if child.status not in ctt.Status.Group.SUCCEED:
                raise common.errors.TaskStop(
                    'One or more child tasks are failed, see "Child tasks" tab for details.')

    def on_execute(self):
        try:
            # Mark build stage as completed only if it exits witout errors. This allows restarting task from sandbox UI.
            with self.memoize_stage.build_stage(commit_on_entrance=False):
                self.build_stage()
        finally:
            # Check if this task is master task.
            if not self.parent:
                with self.memoize_stage.wait_child_tasks_done(commit_on_entrance=False):
                    self.wait_child_tasks_done()

                # Make teamcity resource.
                self.make_teamcity_resource()

                with self.memoize_stage.final_stage(commit_on_entrance=False):
                    self.raise_error_if_child_task_failed()

    def build_stage(self):
        output_dir = os.path.join(self.path_abro(), 'out')
        try:
            # Checkout and build.
            logging.info('Start  [Checkout]')
            self.step_checkout(self.Parameters.branch, self.Parameters.commit,
                               self.path_abro())
            logging.info('Finish [Checkout]')

            logging.info('Start  [GClient]')
            self.step_gclient()
            logging.info('Finish [GClient]')

            try:
                logging.info('Start  [Build]')
                if self.Parameters.yaml_content:
                    self.step_save_yaml(self.Parameters.yaml_path, self.Parameters.yaml_content)
                self.step_build(params=self.Parameters, output_dir=output_dir)
                logging.info('Finish [Build]')
            finally:
                # No need to publish output resource if this task is master.
                if self.parent:
                    self.make_gradle_daemon_resource()
                    self.make_output_resource(self.path_abro())
        except sdk2.WaitTask:
            raise
        except:
            logging.exception('Build stage failed.')
            shutil.rmtree(self.path_abro(), ignore_errors=True)
            raise

    def make_gradle_daemon_resource(self):
        # Make resource and it's dir.
        logging.info('Making gradle daemon resource.')
        res = GradleDaemonResource(self, 'Gradle daemon contents', 'gradle_daemon_resource')
        res_data = sdk2.ResourceData(res)
        res_data.path.mkdir(0o755, parents=True)

        # Copy contents of ~/.gradle/daemon
        src_dir = os.path.expanduser('/cache/gradle/daemon')
        copy_into_dir(src_dir, res_data.path)
        # Recursively change mode to 0777.
        for entry in res_data.path.glob('**/*'):
            entry.chmod(0777)

        # Mark resource's data as 'ready'.
        res_data.ready()

    def make_output_resource(self, checkout_dir):
        checkout_dir = Path(checkout_dir)
        # Rules to selectively copy checkout dir contents into output resource.
        copy_rules = [
            ('.unpackedCores/*/*/*/armRelease/misc/*source_deps.json', 'source_deps/arm-release'),
            ('.unpackedCores/*/*/*/x86Release/misc/*source_deps.json', 'source_deps/arm-release'),
            ('.unpackedCores/*/*/*/armRelease/misc/grit_translations_dump/*', 'grit_dumps/arm-release'),
            ('.unpackedCores/*/*/*/x86Release/misc/grit_translations_dump/*', 'grit_dumps/x86-release'),
            ('browser/app/build/reports/dependencies*', 'deps'),
            ('browser/buildSrc/build/reports/BuildSrc-report.zip', 'reports'),
        ]
        copy_rules.append(('out', '.'))
        # Custom rules for allure reports.
        # Copy contents of dir like 'deps/func_test_runner/target/578fb46c8895507f4dfe2a63-Aphone/allure-report'
        # into target dir like 'allure-report/578fb46c8895507f4dfe2a63-Aphone'.
        for report_dir in checkout_dir.glob('deps/func_test_runner/target/*-Ap*'):
            report_name = report_dir.name
            copy_rules.append((
                str(report_dir.joinpath('allure-report').relative_to(checkout_dir)),
                os.path.join('allure-report', report_name)))

        # Log copy rules.
        logging.info('Output resource copy rules:')
        for src_glob, dst_dir in copy_rules:
            logging.info('"%s" => "%s"', src_glob, dst_dir)

        # Make resource and it's dir.
        logging.info('Making task output resource.')
        res = OutputResource(self, 'Task output', 'output_resource')
        res_data = sdk2.ResourceData(res)
        res_data.path.mkdir(0o755, parents=True)

        # Apply copy rules.
        for src_glob, dst_dir in copy_rules:
            dst_dir = res_data.path.joinpath(dst_dir)
            entries = list(checkout_dir.glob(src_glob))
            if not entries:
                continue
            dst_dir.mkdir(parents=True, exist_ok=True)
            for entry in entries:
                logging.debug('Processing copy rule: "%s" into "%s"', str(entry), str(dst_dir))
                merge_copy(entry, dst_dir)

        # Mark resource's data as 'ready'.
        res_data.ready()

    def make_teamcity_resource(self):
        """
        Combine 'output' resources from each child task all together into single teamcity resource.
        """
        # Make teamcity log resource with service messages. This will be read and executed on TC by resource loader.
        # Log resource MUST be created before artifact resource. That's because TC will process all resources in
        # reverse order (BYIN-5525).
        logging.info('Making teamcity log resource')
        tc_log_res = TeamcityServiceMessagesLog(self, 'Teamcity log', 'teamcity.log', ttl=3)
        tc_log_data = sdk2.ResourceData(tc_log_res)

        # Make resource for teamcity artifacts.
        logging.info('Making teamcity resource')
        tc_data_res = TeamcityArtifacts(self, 'Teamcity artifacts', 'teamcity_artifacts', ttl=365)
        tc_data = sdk2.ResourceData(tc_data_res)
        tc_data.path.mkdir(0o755, parents=True)

        # Iterate over child task 'output' resources and merge them into single dir.
        for child_id in self.Context.child_tasks:
            child_task = self.find(id=child_id).first()
            for out_res in OutputResource.find(task=child_task).limit(100):
                # Synchronize resource data to local disk.
                out_data = sdk2.ResourceData(out_res)
                merge_copy(out_data.path, tc_data.path)

        # Make teamcity build update its build number and publish downloaded artifact.
        tc_log_data.path.write_text(
            u"##teamcity[buildNumber '{}']\n".format(self.Context.yandex_version) +
            u"##teamcity[publishArtifacts 'teamcity_artifacts => .']\n")

        # Mark resources data as ready.
        try:
            tc_log_data.ready()
            tc_data.ready()
        except common.errors.InvalidResource as e:
            logging.error('Error making teamcity resource: %s', str(e))

    def start_single_target(self, target):
        yandex_version = self.Context.yandex_version
        build_defines = list(self.Parameters.build_defines)
        build_defines.append('yandex_version={}'.format(yandex_version))

        params = {
            'kill_timeout': self.Parameters.kill_timeout,
            '_container': self.Parameters._container,
            'branch': self.Parameters.branch,
            'commit': self.Parameters.commit,
            'yaml_path': self.Parameters.yaml_path,
            'yaml_content': self.Parameters.yaml_content,
            'dogfood_branch_name': self.Parameters.dogfood_branch_name,
            'build_defines': build_defines,
            'phone_clids_xml': self.Parameters.phone_clids_xml,
            'tablet_clids_xml': self.Parameters.tablet_clids_xml,
            'single_target': target,
            'parallel_targets': False,
            'teamcity_build_id': self.Parameters.teamcity_build_id,
            'yandex_version': yandex_version,
        }
        child_task = BrowserAndroidBuildTask(
            self,
            description='Target: {}'.format(target),
            priority=min(
                self.Parameters.priority,
                # default API limitation
                ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.NORMAL)
            ),
            **params
        )
        child_task.Requirements.tasks_resource = self.Requirements.tasks_resource
        child_task.save().enqueue()
        return child_task
