import fnmatch
from functools import partial
import os
import logging
import re
import shutil
import subprocess
from xml.etree import ElementTree

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox.sandboxsdk.environments import SandboxEnvironment, VirtualEnvironment
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.vcs.git import Git


def clids_from_xml_string(xml_string):
    """
    Parse clids from XML string like this:
        <?xml version="1.0" encoding="windows-1251"?>
        <vendor name="browser-ym-android-tablet.ya.ru_348327" setup="bar-only">
        <clid1>2066559</clid1>
        ...
        <clid1010>2263191</clid1010>
        </vendor>

    :param xml_string: str with XML contents.
    :return: dict[clid_name: clid_value]
    """
    clids = {}
    vendor_node = ElementTree.fromstring(xml_string)
    assert vendor_node.tag == 'vendor'
    for clid_node in vendor_node:
        assert clid_node.tag.startswith('clid')
        clids[clid_node.tag] = clid_node.text
    return clids


class BrowserAndroidApkResource(sdk2.Resource):
    """ browser-android APK """

    yaml = sdk2.parameters.String()
    version = sdk2.parameters.String()


class BrowserAndroidTaskBase(sdk2.Task):
    """ Base class for BrowserAndroid tasks. """

    REPO_NAME = 'browser-android'
    VCS_URL = 'https://bitbucket.browser.yandex-team.ru/scm/stardust/browser-android.git'

    class Context(sdk2.Task.Context):
        yandex_version = None

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.BROWSER & ctc.Tag.LINUX_TRUSTY
        dns = ctm.DnsType.DNS64
        disk_space = (48 * 1024)
        ram = (8 * 1024)

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        kill_timeout = 18000  # 5h

        with sdk2.parameters.Group("Miscellaneous Parameters") as misc_block:
            _container = sdk2.parameters.Container("Container", default=None, required=True)

        with sdk2.parameters.Group('Credentials') as credentials_group:
            yav_token_vault = sdk2.parameters.String(
                'Vault item with yav token', default='robot-browser-infra_yav_token')

    def path_abro(self):
        return str(self.path(self.REPO_NAME))

    def step_checkout(self, branch, commit, checkout_dir):
        subprocess.check_call(['git', 'config', '--global', 'user.email', 'teamcity@yandex-team.ru'])
        subprocess.check_call(['git', 'config', '--global', 'user.name', 'Teamcity'])
        vcs_root = Git(self.VCS_URL, filter_branches=False)
        vcs_root.clone(checkout_dir, branch=branch, commit=commit)

    def step_gclient(self):
        gclient = '/tools/depot_tools/gclient'
        gclient_cache_dir = '/cache/task/gclient'

        gclient_env = os.environ
        gclient_env['PATH'] = gclient_env['PATH'] + ':/tools/depot_tools/'
        gclient_env['DOWNLOAD_EMULATOR_SDK'] = '1'

        with sdk2.helpers.ProcessLog(self, logger='gclient') as pl:
            sp.check_call(' '.join([
                gclient, 'config',
                '--deps-file', 'deps.config',
                '--name', self.path_abro(),
                '--unmanaged', self.VCS_URL,
                '--cache-dir', gclient_cache_dir,
            ]), shell=True, cwd=self.path_abro(), env=gclient_env, stdout=pl.stdout, stderr=sp.STDOUT)
            sp.check_call(' '.join([
                gclient, 'sync',
                '--reset', '--force', '--verbose', '--ignore_locks',
            ]), shell=True, cwd=self.path_abro(), env=gclient_env, stdout=pl.stdout, stderr=sp.STDOUT)

    def step_save_yaml(self, yaml_path, content):
        yaml_abspath = os.path.join(self.path_abro(), yaml_path)
        yaml_dir = os.path.dirname(yaml_abspath)
        if not os.path.isdir(yaml_dir):
            os.makedirs(yaml_dir)
        with open(yaml_abspath, 'w') as yaml_file:
            yaml_file.write(content)

    def start_single_target(self, target):
        logging.error('Not implemented')
        pass

    def init_teamcity_build_id(self):
        os.environ['TEAMCITY_BUILD_ID'] = self.Parameters.teamcity_build_id

    def step_build(self, params, output_dir):
        self.init_teamcity_build_id()

        branch_name = params.branch[len('refs/heads/'):]
        with VirtualEnvironment(use_system=True) as virtualenv:
            # This should work fine even without retries. Since VirtualEnvironment.pip has fallback to pypi.python.org
            # in case of errors with pypi.yandex-team.ru.
            virtualenv.pip('pip==9.0.3')
            virtualenv.pip('setuptools==39.0.1')
            virtualenv.pip('retry==0.9.2')

            # Here we're defenitely need to install from pypi.yandex-team.ru. Since theese requirements.txt files
            # contain internal packages. Use retry to make this stable (pypi.yandex-team.ru responds 502 sometimes).
            activate_this = os.path.join(virtualenv.root_dir, 'bin', 'activate_this.py')
            execfile(activate_this, dict(__file__=activate_this))
            from retry.api import retry_call
            retry_call(
                partial(virtualenv.pip, '-r ' + os.path.join(self.path_abro(), 'requirements.txt')),
                tries=8, delay=2, backoff=2)
            retry_call(
                partial(virtualenv.pip, '-r ' + os.path.join(self.path_abro(), 'deps/yin/requirements.txt')),
                tries=8, delay=2, backoff=2)

            os.environ['GRADLE_USER_HOME'] = SandboxEnvironment.exclusive_build_cache_dir('gradle')
            os.environ['MAVEN_HOME'] = '/usr/share/maven-3.3.9'
            os.environ['JAVA8_HOME'] = '/usr/lib/jvm/java-8-oracle'
            # os.environ['NEED_ACCEPTANCE'] = 'YES'
            os.environ['NEED_SMOKE'] = 'YES'
            os.environ['SMOKE_REPORT_CHANNEL'] = '#abro_release_reports'
            os.environ['EMULATED_RATE'] = '100'
            os.environ['EMULATED'] = 'YES'
            os.environ['TESTSUITE_TIMEOUT'] = '120'
            os.environ['TESTPACKS_TO_RUN'] = '578fb4518895507f4dfe2a60,578fb46c8895507f4dfe2a63'
            os.environ['SLACKER_TOKEN'] = sdk2.Vault.data('ABRO_SLACK_TOKEN')
            os.environ['ST_TOKEN'] = sdk2.Vault.data('ABRO_ST_TOKEN')
            os.environ['TC_USERNAME'] = 'robot-mbro-infra'
            os.environ['TC_PASSWORD'] = sdk2.Vault.data('ROBOT_MBRO_INFRA_PASSWORD')
            os.environ['SANDBOX_TOKEN'] = sdk2.Vault.data('ABRO_SANDBOX_TOKEN')
            os.environ['YAV_TOKEN'] = sdk2.Vault.data(self.Parameters.yav_token_vault)
            os.environ['NEED_STATFACE_REPORT'] = 'YES'
            os.environ['NEED_TESTPALM_REPORT'] = 'YES'
            os.environ['ENABLE_SLACK_COMMENTS'] = 'YES'
            os.environ['PROXY_PRVKEY'] = sdk2.Vault.data('ROBOT_MBRO_INFRA_SSH_PRIVATE_KEY')

            os.environ['PATH'] = os.environ['PATH'] + ':' + os.environ['JAVA8_HOME'] + '/bin'

            os.environ['SANDBOX_TASK_ID'] = '{}'.format(self.id)

            if not os.path.isdir(os.path.expanduser('~/.yandex')):
                os.mkdir(os.path.expanduser('~/.yandex'))
            signer_token = sdk2.Vault.data('ABRO_SIGNER_OAUTH_TOKEN')
            with open(os.path.expanduser('~/.yandex/yandex-signer.properties'), 'w') as signer_file:
                signer_file.write('oauth={}'.format(signer_token))

            yaml_file = os.path.join(self.path_abro(), params.yaml_path)
            if params.parallel_targets:
                with sdk2.helpers.ProcessLog(self, logger="targets") as pl:
                    # get list of targets
                    get_targets_script = (
                        "\""
                        "import sys;"
                        "sys.path.append('" + os.path.join(self.path_abro(), 'build') + "');"
                        "import build;"
                        "config = build.make_config('" + yaml_file + "', []);"
                        "print '\\n'.join([target['name'] for target in config.Get('targets')])"
                        "\"")
                    command = [virtualenv.executable, '-c', get_targets_script]
                    output = sp.check_output(' '.join(command), shell=True, cwd=self.path_abro())
                    target_list = output.split()

                    # Get Yandex version and put it into Context.
                    get_version_script = (
                        "\""
                        "import sys;"
                        "sys.path.append('" + os.path.join(self.path_abro(), 'build') + "');"
                        "from steps import initgradle;"
                        "version = initgradle.get_default_yandex_version();"
                        "version = initgradle.inc_yandex_version(version);"
                        "print version"
                        "\"")
                    command = [virtualenv.executable, '-c', get_version_script]
                    yandex_version = sp.check_output(' '.join(command), shell=True, cwd=self.path_abro())
                    yandex_version = yandex_version.strip()
                    self.Context.yandex_version = yandex_version

                    child_tasks = []
                    for target in target_list:
                        logging.info('Starting child task for target: %s', str(target))
                        child_tasks.append(self.start_single_target(target))

                    self.Context.child_tasks = map(lambda task: task.id, child_tasks)
                    raise sdk2.WaitTask(child_tasks, (ctt.Status.Group.FINISH, ctt.Status.Group.BREAK))
            else:
                with sdk2.helpers.ProcessLog(self, logger="build") as pl:
                    command = [
                        virtualenv.executable, os.path.join(self.path_abro(), 'teamcity.py'),
                        '--config', yaml_file,
                        '--out-dir', output_dir,
                        '--environment', 'local',
                        '--branch', branch_name,
                    ]
                    if params.dogfood_branch_name:
                        command.extend(['--dogfood-branch', params.dogfood_branch_name])
                    build_defines = list(params.build_defines)
                    if params.single_target:
                        build_defines.append('build_single_target=' + params.single_target)
                    if params.phone_clids_xml:
                        for clid_name, clid_value in clids_from_xml_string(params.phone_clids_xml).iteritems():
                            build_defines.append('config.{}_phone={}'.format(clid_name, clid_value))
                    if params.tablet_clids_xml:
                        for clid_name, clid_value in clids_from_xml_string(params.tablet_clids_xml).iteritems():
                            build_defines.append('config.{}_tablet={}'.format(clid_name, clid_value))
                    for build_define in build_defines:
                        if build_define:
                            command.extend(['-D', build_define])

                    try:
                        # Build browser
                        sp.check_call(
                            ' '.join(command),
                            shell=True, cwd=self.path_abro(), stdout=pl.stdout, stderr=sp.STDOUT
                        )
                    finally:
                        # Print output directory contents.
                        with sdk2.helpers.ProcessLog(self, logger="output") as pl_output:
                            sp.check_call('find ' + os.path.abspath(output_dir),
                                          shell=True, stdout=pl_output.stdout, stderr=sp.STDOUT)

                        # Publish APKs
                        apk_resources = []
                        for root, dirnames, filenames in os.walk(os.path.join(self.path_abro(), 'browser/app/build')):
                            for filename in fnmatch.filter(filenames, 'app-*-release.apk'):
                                apk_resources.append(os.path.join(root, filename))
                        logging.info('APK artifacts : {}'.format(apk_resources))
                        for apk_resource_file in apk_resources:
                            logging.info('Publish APK : {}'.format(apk_resource_file))

                            # Generate new APK name.
                            yandex_version = self.Context.yandex_version
                            new_resource_name = os.path.basename(apk_resource_file)
                            apk_version_parts = yandex_version.split('.')
                            apk_prefix = '%02d%02d%01d%04d' % tuple([int(p) for p in apk_version_parts])
                            if re.match('.*arm.*api16.*', new_resource_name):
                                apk_prefix += '0'
                            elif re.match('.*x86.*api16.*', new_resource_name):
                                apk_prefix += '1'
                            elif re.match('.*arm.*api21.*', new_resource_name):
                                apk_prefix += '2'
                            elif re.match('.*x86.*api21.*', new_resource_name):
                                apk_prefix += '3'
                            elif re.match('.*arm.*api23.*', new_resource_name):
                                apk_prefix += '4'
                            else:
                                apk_prefix += 'X'
                                logging.error('Strange APK variant: {}'.format(new_resource_name))
                            new_resource_name = re.sub('^app', apk_prefix + '-yandexbrowser', new_resource_name)

                            # Copy APK with new name.
                            new_resource_file = os.path.join(os.path.dirname(apk_resource_file), new_resource_name)
                            if apk_resource_file != new_resource_file:
                                shutil.copyfile(apk_resource_file, new_resource_file)

                            # Publish resource.
                            resource = BrowserAndroidApkResource(self, new_resource_name, new_resource_file)
                            resource.yaml = params.yaml_path
                            resource.version = yandex_version
                            data = sdk2.ResourceData(resource)
                            data.ready()
