"""
Sandbox task that provides implementation of sandbox-lite-api.

See: https://bitbucket.browser.yandex-team.ru/projects/MBROINFRA/repos/sandbox-integration/
See: https://testenv.yandex-team.ru/?screen=jobs&database=sandbox_lite_tests
"""

import json
import logging
import os
import re
import subprocess
import textwrap

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox import sandboxsdk
from sandbox import sdk2
from sandbox.sandboxsdk.environments import SandboxEnvironment


class LiteResource(sdk2.Resource):
    """
    Generic resource to be created and consumed via
    sandbox-lite-api users.
    """
    lite_name = sdk2.Attributes.String('Resource name', required=True)
    calc_md5 = False
    share = False
    ttl = 1


class BrowserAndroidSandboxLiteTask(sdk2.Task):
    """ Provides implementation of sandbox-lite-api """

    __PIP_INDEX_URL_REGEX = r'--index-url\s*(.*)'

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.BROWSER & ctc.Tag.LINUX_TRUSTY
        dns = ctm.DnsType.DNS64
        disk_space = 48 * 1024
        ram = 8 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):

        teamcity_build_id = sdk2.parameters.String(
            'ID of TeamCity build triggered this task.')
        container = sdk2.parameters.Container(
            'LXC container to emulate environment.',
            default=None)
        tasks_archive = sdk2.parameters.Resource(
            'Resource containing tasks archive to use.',
            default=None)
        requirements = sdk2.parameters.String(
            'Python requirements to install.',
            multiline=True,
            required=True)
        task_git_url = sdk2.parameters.String(
            'Task git URL.')
        task_git_branch = sdk2.parameters.String(
            'Task git branch.')
        task_git_commit = sdk2.parameters.String(
            'Task git commit.')
        task_requirements_paths = sdk2.parameters.String(
            'Task requirements paths in git repository separated by commas.')
        task_requirements_text = sdk2.parameters.String(
            'Task requirements text.',
            multiline=True)
        task_import_paths = sdk2.parameters.String(
            'Task import paths in git repository separated by commas.')
        task_class = sdk2.parameters.String(
            'Task class name.')
        task_code = sdk2.parameters.String(
            'Task code.',
            multiline=True)
        with sdk2.parameters.Group('Task parameters'):
            task_params_dict = sdk2.parameters.Dict('Task parameters.')
            task_params_json = sdk2.parameters.JSON('Task parameters.')
            launcher_context_params = sdk2.parameters.Dict(
                'Parameters from launcher context. Please do not edit.')

    def on_save(self):
        if self.Parameters.tasks_archive is not None:
            self.Requirements.tasks_resource = self.Parameters.tasks_archive

        launcher_context = self.Context.sandbox_task_launcher_context
        if launcher_context != ctm.NotExists:
            self.Parameters.launcher_context_params = dict(launcher_context)

        if self.Parameters.launcher_context_params:
            for k, v in self.Parameters.launcher_context_params.iteritems():
                self.Parameters.task_params_dict.setdefault(k, v)

    def install_requirements(self):
        requirements_text = str(self.Parameters.requirements)
        if requirements_text:
            # Find index URL.
            matcher = re.search(self.__PIP_INDEX_URL_REGEX, requirements_text)
            index_url = matcher.group(1) if matcher else None

            # Install modules.
            for module_name in requirements_text.split('\n'):
                # Skip empty strings.
                if not module_name.strip():
                    continue

                # Skip index URL.
                if re.search(self.__PIP_INDEX_URL_REGEX, module_name):
                    continue

                # Install using PipEnvironment.
                sandboxsdk.environments.PipEnvironment(
                    module_name, index_url=index_url).prepare()

    def init_environment(self):
        """
        Initializes environment in LXC-container.

        When task is executed in LXC-container it doesn't init
        environment. So we should programmatically get environment
        from /etc/profile and ~/.bashrc
        """
        # Skip initialization if container not set.
        if not self.Parameters.container:
            return

        environ_json = subprocess.check_output([
            'bash', '-c', textwrap.dedent('''
                for script in "$@"; do
                    . "$script"
                done
                python -c """
                import json
                import os
                print(json.dumps(dict(os.environ)))
                """
            '''), 'script.sh',  # Command line argument "$0".
            '/etc/profile', os.path.expanduser('~/.bashrc')])
        os.environ.clear()
        os.environ.update(json.loads(environ_json))

    def do_create_task_paths(self, cache_dir):
        from sandbox_lite_task import implementation
        return implementation.SandboxTaskPaths(self, cache_dir)

    def get_cache_dir(self):
        return SandboxEnvironment.exclusive_build_cache_dir('sandbox-lite-cache')

    def create_task_paths(self):
        cache_dir = self.get_cache_dir()
        logging.info('Cache directory: %s', cache_dir)
        return self.do_create_task_paths(cache_dir)

    def create_task_helper(self):
        from sandbox_lite_task import helper
        return helper.DefaultTaskHelper(self, self.create_task_paths())

    def get_task_environ(self):
        return {}

    def on_execute(self):
        self.install_requirements()
        self.init_environment()
        self.create_task_helper().on_execute(self.get_task_environ())
