import contextlib
import json
import re
from functools import partial
import os
import shutil
import tarfile

import pathlib2
import yaml

from sandbox import sdk2
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
from sandbox.common.errors import TaskFailure
import sandbox.projects.common.arcadia.sdk as arcadia_sdk
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment, SandboxGitLfsEnvironment
from sandbox.projects.common.teamcity import TeamcityArtifacts, TeamcityArtifactsContext, TeamcityServiceMessagesLog
from sandbox.projects.browser.common.git.repositories import bitbucket_vcs_root
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.sdk2.helpers import subprocess
from sandbox.sdk2.environments import SandboxEnvironment

VAULT_PLACEHOLDER_RE = re.compile(r'{vault\.(.*?)}')
STRING_PARAMETERS_SEPARATOR = ' '
STRING_PARAMETERS_DESCRIPTION = 'Separated by "{}"'.format(STRING_PARAMETERS_SEPARATOR)
CONFIG_SCHEMA = os.path.join(os.path.abspath(os.path.dirname(__file__)), 'launching_config_schema.json')


def validate_schema(config):
    import jsonschema
    with open(CONFIG_SCHEMA) as f:
        schema = json.load(f)
        try:
            jsonschema.validate(config, schema)
        except jsonschema.exceptions.ValidationError as e:
            raise TaskFailure("Incorrect launching config: {}".format(e.message))


def split_param(param):
    return [v for v in param.split(STRING_PARAMETERS_SEPARATOR) if v]


class CommonParameters(sdk2.Parameters):
    with sdk2.parameters.Group('Scm') as scm:
        with sdk2.parameters.String("VCS") as vcs:
            vcs.values.git = vcs.Value("git", default=True)
            vcs.values.arc = vcs.Value("arc")

        with vcs.value['git']:
            project = sdk2.parameters.String('Project', required=True)
            repo = sdk2.parameters.String('Repository', required=True)
            branch = sdk2.parameters.String('Branch')
            commit = sdk2.parameters.String('Commit', required=True)

        with vcs.value['arc']:
            arcadia_dir = sdk2.parameters.String('Arcadia dir', required=True)
            arcadia_ref = sdk2.parameters.String('Arcadia ref', required=True, default='trunk')

    jdk_version = sdk2.parameters.String('JDK version', default='1.8.0')
    build_sdk = sdk2.parameters.Resource('Android sdk', required=True, default=3350904867)
    gradle_targets = sdk2.parameters.String(
        'Gradle targets', description=STRING_PARAMETERS_DESCRIPTION, required=True)
    gradle_arguments = sdk2.parameters.String(
        'Gradle arguments', default='',
        description=STRING_PARAMETERS_DESCRIPTION + '. Use "{vault.<vault_item_name>}" to pass secrets from vault.'
                                                    'e.g. "-Ppassword={vault.some_secret}"', )
    emulator_launching_config = sdk2.parameters.Resource('Emulator launching config',
                                                         required=True, default=1075024142)
    artifacts_to_publish = sdk2.parameters.String(
        'Artifacts to publish',
        description='Files and directories, which will be added to TeamcityArtifacts resource. {}'.format(
            STRING_PARAMETERS_DESCRIPTION))


class BrowserRunAndroidUiTests(sdk2.Task):
    class Parameters(CommonParameters):
        launch_name = sdk2.parameters.String('Launch name',  default='default_config', required=True)
        debug = sdk2.parameters.Bool('Debug', default=False)
        _container = sdk2.parameters.Container("Container", default=890998880, required=True)

    class Requirements(sdk2.Task.Requirements):
        disk_space = 10 * 1024  # 10GB
        client_tags = ctc.Tag.BROWSER & ctc.Tag.LINUX_TRUSTY
        cores = 16
        dns = ctm.DnsType.DNS64
        environments = [
            SandboxGitLfsEnvironment('2.5.2'),
            PipEnvironment('decorator==4.4.2'),
            PipEnvironment('get-deps==1.2.3'),
            PipEnvironment('jsonschema==2.5.1'),
            # Here is a workaround to install avd tools without PyYAML
            # We can't install avd tools to user site, because we have a conflict
            # in dependencies.
            #
            # InstallationError: Will not install to the user site because it will lack
            # sys.path precedence to PyYAML in ...
            #
            # First we install avd tools without any dependencies and then
            # we install all dependencies except PyYAML.
            PipEnvironment('yandex-avd-tools==1.4.0',
                           custom_parameters=['--no-deps']),
            PipEnvironment('decorator>=4.4.2,<5'),
            PipEnvironment('killall>=1.1.0,<2'),
            PipEnvironment('py>=1.5.2,<2'),
            PipEnvironment('pyjavaproperties>=0.7,<1'),
            PipEnvironment('retry>=0.9.2,<1'),
            PipEnvironment('six<2'),
        ]
        ram = 31 * 1024

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    def on_prepare(self):
        jdk_version = self.Parameters.jdk_version
        SandboxJavaJdkEnvironment(version=jdk_version, platform='linux').prepare()
        super(BrowserRunAndroidUiTests, self).on_prepare()

    def artifacts_path(self, *args):
        return str(self.path('teamcity_artifacts', *args))

    def build_sdk_path(self, *args):
        return str(self.path('build-sdk', *args))

    def emulator_sdk_path(self, *args):
        return str(self.path('emulator-sdk', *args))

    def device_log_path(self, project_root):
        return os.path.join(project_root, 'logcat.txt')

    @contextlib.contextmanager
    def checkout_code(self):
        if self.Parameters.vcs == 'git':
            checkout_dir = str(self.path('checkout-dir'))
            vcs_root = partial(bitbucket_vcs_root, self.Parameters.project, self.Parameters.repo)(filter_branches=False)
            vcs_root.clone(checkout_dir, self.Parameters.branch, commit=self.Parameters.commit)
            vcs_root.execute('submodule', 'update', '--init', '--recursive', cwd=checkout_dir)
            yield checkout_dir
        elif self.Parameters.vcs == 'arc':
            # https://st.yandex-team.ru/ARC-3966
            os.environ['ARC_SKIP_SERVER_CACHE'] = 'true'
            with arcadia_sdk.mount_arc_path('arcadia-arc:/#{}'.format(self.Parameters.arcadia_ref)) as arc_root:
                yield os.path.join(arc_root, self.Parameters.arcadia_dir)
        else:
            assert False, 'incorrect vcs type'

    def get_build_sdk(self):
        archive_path = str(sdk2.ResourceData(self.Parameters.build_sdk).path)
        with tarfile.open(archive_path, 'r') as archive:
            archive.extractall(path=self.build_sdk_path())

    def get_emulators_sdk(self, emulator_sdk_url, emulator_sdk_hash):
        from get_deps import get_deps
        os.mkdir(self.emulator_sdk_path())
        cache_dir = os.path.join(SandboxEnvironment.build_cache_dir, 'get-deps')
        getdeps = get_deps.GetDeps(cache_dir=cache_dir)
        getdeps.deploy(
            url=emulator_sdk_url,
            deploy_rules=[('copy', '*', self.emulator_sdk_path() + '/')],
            sha1=emulator_sdk_hash,
        )

    def gradle(self, project_root):
        gradle_args = (split_param(self.Parameters.gradle_arguments)
                       if self.Parameters.gradle_arguments else [])
        gradle_args = [
            re.sub(VAULT_PLACEHOLDER_RE, lambda match: sdk2.Vault.data(match.group(1)), arg)
            for arg in gradle_args
        ]
        with TeamcityArtifactsContext(pathlib2.Path(project_root)) as artifacts_context:
            env = os.environ.copy()
            env['ANDROID_HOME'] = self.build_sdk_path()
            env['EMULATOR_SDK_PATH'] = self.emulator_sdk_path()
            env['GRADLE_USER_HOME'] = SandboxEnvironment.exclusive_build_cache_dir('gradle')
            return subprocess.call(
                ['./gradlew'] + gradle_args + split_param(self.Parameters.gradle_targets),
                cwd=project_root, env=env,
                stdout=artifacts_context.output, stderr=artifacts_context.output)

    def publish_artifacts(self, files_to_publish, project_root):
        publish = False
        os.mkdir(self.artifacts_path())
        for filename in files_to_publish:
            abs_path = os.path.abspath(os.path.join(project_root, filename))
            if not os.path.exists(abs_path):
                continue
            if os.path.isdir(abs_path):
                shutil.copytree(abs_path, self.artifacts_path(os.path.basename(abs_path)))
            else:
                shutil.copy(abs_path, self.artifacts_path())
        for root, dirs, files in os.walk(self.artifacts_path()):
            if files:
                publish = True
                break
        if publish:
            artifacts_resource = TeamcityArtifacts(
                self, 'Teamcity artifacts', self.artifacts_path())
            sdk2.ResourceData(artifacts_resource).ready()

    @contextlib.contextmanager
    def start_logcat(self, log_file):
        from avd_tools import paths
        logcat_process = None
        try:
            logcat_process = subprocess.Popen(
                [paths.adb_path(), 'logcat'], stdout=log_file)
            yield logcat_process
        finally:
            if logcat_process is not None:
                logcat_process.kill()

    @sdk2.header()
    def header(self):
        result = []

        log_resource = TeamcityServiceMessagesLog.find(task=self).first()
        artifacts_resource = TeamcityArtifacts.find(task=self).first()

        if log_resource:
            result.append('<a href="{}">Execution log</a>'.format(
                log_resource.http_proxy,
            ))

        artifacts = split_param(self.Parameters.artifacts_to_publish)
        if artifacts_resource:
            for artifact_path in artifacts:
                result.append('<a href="{}/{}">{}</a>'.format(
                    artifacts_resource.http_proxy, os.path.basename(artifact_path), artifact_path,
                ))

        return '<br />'.join(result)

    def on_execute(self):
        import avd_tools

        launch_name = self.Parameters.launch_name or 'default_config'
        with sdk2.ResourceData(self.Parameters.emulator_launching_config).path.open() as config:
            emulator_configs = yaml.load(config)
        validate_schema(emulator_configs)
        if launch_name not in emulator_configs:
            raise TaskFailure('Can\'t find launch name {} in config'.format(launch_name))
        emulator_sdk_url = emulator_configs[launch_name]['sdk_url']
        emulator_sdk_hash = emulator_configs[launch_name]['sdk_hash']

        with self.checkout_code() as project_root:
            self.get_build_sdk()
            self.get_emulators_sdk(emulator_sdk_url, emulator_sdk_hash)
            avd_tools.utils.kill_stale_processes()
            avd_tools.utils.remove_android_home_dir()
            avd_tools.paths.set_sdk_path(self.emulator_sdk_path())
            avd_tools.paths.set_adb_path(self.build_sdk_path('platform-tools', 'adb'))
            with open(os.path.abspath(self.device_log_path(project_root)), 'w') as logcat_out:
                try:
                    emulator_launching_config_path = str(self.path('{}.yaml'.format(launch_name)))
                    with open(emulator_launching_config_path, 'w') as launching_config_file:
                        yaml.dump(emulator_configs[launch_name]['launching_config'], launching_config_file)
                    with avd_tools.RunManyAvdWithRetry(emulator_launching_config_path, tries=3):
                        with self.start_logcat(logcat_out):
                            if self.Parameters.debug:
                                self.suspend()
                            exit_code = self.gradle(project_root)
                            if exit_code:
                                raise TaskFailure('Gradle exited with {}'.format(exit_code))
                finally:
                    self.publish_artifacts(split_param(self.Parameters.artifacts_to_publish) + [self.device_log_path(project_root)], project_root)
