import os
import shutil

import yaml

import sandbox
from sandbox.common.errors import TaskFailure
from sandbox.common.types.resource import State as ResourceState
from sandbox.common.types.task import Status
from sandbox.projects.common.teamcity import TeamcityArtifacts, TeamcityServiceMessagesLog
from sandbox.projects.browser_mobile.android.BrowserRunAndroidUiTests import (
    BrowserRunAndroidUiTests, CommonParameters, validate_schema, STRING_PARAMETERS_DESCRIPTION, split_param)
from sandbox import sdk2
import sandbox.common.types.client as ctc
from sandbox.sandboxsdk.environments import PipEnvironment


class BrowserRunMultipleAndroidUiTests(sdk2.Task):
    class Parameters(CommonParameters):
        launch_names = sdk2.parameters.String(
            'Launch names', required=False,
            description='Run only this configurations from launching config. ' + STRING_PARAMETERS_DESCRIPTION)

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.BROWSER
        cores = 1
        environments = [
            PipEnvironment('jsonschema==2.5.1'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(sdk2.Context):
        launched_tasks = []

    def artifacts_path(self, *args):
        return str(self.path('artifacts', *args))

    @sandbox.common.utils.singleton
    def teamcity_log_path(self):
        return str(self.path('teamcity_messages.log'))

    def create_subtask(self, config_name):
        params = dict(self.Parameters)
        params['launch_name'] = config_name
        params['description'] = '[{}] {}'.format(config_name, self.Parameters.description)

        child = BrowserRunAndroidUiTests(self, **params).enqueue()
        self.Context.launched_tasks.append(child.id)

    def create_subtasks(self):
        with self.memoize_stage.run_subtasks:
            with sdk2.ResourceData(self.Parameters.emulator_launching_config).path.open() as config:
                emulator_configs = yaml.load(config)
            validate_schema(emulator_configs)
            if self.Parameters.launch_names:
                launch_names = split_param(self.Parameters.launch_names)
                missings_names = [name for name in launch_names if name not in emulator_configs]
                if missings_names:
                    raise TaskFailure(
                        'This configurations are missing in launching config: {}'.format(', '.join(missings_names)))
                for launch_name in split_param(self.Parameters.launch_names):
                    self.create_subtask(launch_name)
            else:
                for config_name, parameters in emulator_configs.iteritems():
                    self.create_subtask(config_name)

            raise sdk2.WaitTask(
                self.Context.launched_tasks,
                list(Status.Group.FINISH + Status.Group.BREAK),
                True,
            )

    def load_artifacts(self, launch_name, teamcity_artifacts_resource):
        resource_data_path = str(sdk2.ResourceData(teamcity_artifacts_resource).path)
        publish_artifacts = any(files for _, _, files in os.walk(resource_data_path))
        if publish_artifacts:
            shutil.copytree(resource_data_path, os.path.join(self.artifacts_path(), launch_name))
        return publish_artifacts

    def copy_log(self, launch_name, teamcity_messages_resource):
        with open(self.teamcity_log_path(), 'a') as messages_file:
            messages_file.write("##teamcity[blockOpened name='{}']\n".format(launch_name))

            if teamcity_messages_resource.state == ResourceState.READY:
                with sdk2.ResourceData(teamcity_messages_resource).path.open() as log:
                    shutil.copyfileobj(log, messages_file)
            else:
                messages_file.write('Log resource is not in READY state')

            messages_file.write("\n##teamcity[blockClosed name='{}']\n".format(launch_name))

    def add_buildproblem(self, build_problem):
        with open(self.teamcity_log_path(), 'a') as messages_file:
            messages_file.write("##teamcity[buildProblem description='{}']\n".format(build_problem))

    @sdk2.header()
    def header(self):
        result = []

        for subtask in self.find(BrowserRunAndroidUiTests):
            result.append('<h2>{}</h2><br />{}'.format(subtask.Parameters.launch_name, subtask.header()))

        return '<hr />'.join(result)

    def on_execute(self):
        os.mkdir(self.artifacts_path())
        publish_artifacts = False
        failure_reasons = []
        self.create_subtasks()
        for child in self.find():
            teamcity_artifacts_resource = TeamcityArtifacts.find(task=child).first()
            teamcity_messages_resource = TeamcityServiceMessagesLog.find(task=child).first()
            launch_name = child.Parameters.launch_name
            if child.status != Status.SUCCESS:
                message = '{} launch is in status {}'.format(launch_name, child.status)
                failure_reasons.append(message)
                self.add_buildproblem(message)
            if teamcity_artifacts_resource:
                publish_artifacts |= self.load_artifacts(launch_name, teamcity_artifacts_resource)
            if teamcity_messages_resource:
                self.copy_log(launch_name, teamcity_messages_resource)
        if publish_artifacts:
            sdk2.ResourceData(TeamcityArtifacts(self, '', self.artifacts_path())).ready()
        sdk2.ResourceData(TeamcityServiceMessagesLog(self, '', self.teamcity_log_path())).ready()
        if failure_reasons:
            raise TaskFailure('\n'.join(failure_reasons))
