"""
Author: Vladimir Stefanovskiy <stef@yandex-team.ru>
"""
import os

import yaml

from sandbox import sdk2
from sandbox import common
import sandbox.common.types.task as ctt

from sandbox.projects.sandbox import sandbox_lxc_image

SCRIPT_DIR = os.path.dirname(os.path.abspath(__file__))


def available_flavors():
    try:
        filenames = os.listdir(os.path.join(SCRIPT_DIR, 'flavors'))
    except OSError:
        yield 'common'
    else:
        for filename in filenames:
            base, ext = os.path.splitext(filename)
            if ext == '.yaml':
                yield base


class ImageFlavor(sdk2.parameters.String):
    @common.utils.singleton_classproperty
    def default_value(cls):
        return next(available_flavors(), 'common')


class BrowserMobileBuildLxcImage(sdk2.Task):
    """
    Build LXC image described by flavor.
    """

    class Parameters(sdk2.Task.Parameters):
        with ImageFlavor('Image flavor') as flavor_image:
            for flavor in available_flavors():
                flavor_image.values[flavor] = flavor

    class Context(sdk2.Task.Context):
        build_task_id = 0

    @property
    def flavor(self):
        return self.Parameters.flavor_image

    def run_build_task(self):
        flavor_name = self.flavor
        flavor_path = os.path.join(
            SCRIPT_DIR, 'flavors', flavor_name + '.yaml')
        with open(flavor_path, 'r') as f:
            flavor = yaml.load(f)

        image_attrs = {'mbro_lxc_flavor': flavor_name}
        params = {
            sandbox_lxc_image.SandboxLxcImage.Parameters.ubuntu_release.name: flavor['ubuntu_release'],
            sandbox_lxc_image.SandboxLxcImage.Parameters.custom_image.name: True,
            sandbox_lxc_image.SandboxLxcImage.Parameters.test_result_lxc.name: False,
            sandbox_lxc_image.SandboxLxcImage.Parameters.custom_attrs.name: image_attrs,
        }

        if 'custom_scripts' in flavor:
            script = ''
            for script_path in flavor['custom_scripts']:
                script_path = os.path.normpath(os.path.join(
                    SCRIPT_DIR, 'flavors', script_path))
                with open(script_path, 'r') as f:
                    script += f.read() + '\n'
            params[sandbox_lxc_image.SandboxLxcImage.Parameters.custom_script.name] = script

        if 'ubuntu_repos' in flavor:
            params[sandbox_lxc_image.SandboxLxcImage.Parameters.custom_repos.name] = '\n'.join(
                flavor['ubuntu_repos'])

        if 'packages' in flavor:
            params[sandbox_lxc_image.SandboxLxcImage.Parameters.custom_packages.name] = ' '.join(
                flavor['packages'])

        build_task = sandbox_lxc_image.SandboxLxcImage(
            self,
            description=flavor['description'],
            priority=min(
                self.Parameters.priority,
                ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.NORMAL)  # default API limitation
            ),
            **params
        )
        self.Context.build_task_id = build_task.enqueue().id

        raise sdk2.WaitTask(
            self.Context.build_task_id,
            tuple(ctt.Status.Group.FINISH) + tuple(ctt.Status.Group.BREAK)
        )

    def on_execute(self):
        with self.memoize_stage.fire_once:
            self.run_build_task()

        build_task = sdk2.Task[self.Context.build_task_id]
        if build_task.status != ctt.Status.SUCCESS:
            raise common.errors.TaskFailure(
                'Build image task failed, see child task for details')
