# Stop on first error.
set -e


# Downloads Sandbox resource.
# Usage: <resource_id> <target_path>
#     resource_id - sandbox resource ID.
#     target_path - path to save resource.
download_sandbox_resource() {
    resource_id="$1"
    target_path="$2"
    wget "https://proxy.sandbox.yandex-team.ru/$resource_id" -O "$target_path"
}


# Unpacks archive into specified directory.
# Usage: <archive_path> <target_path> <args>*
#     archive_path - path of archive to unpack.
#     target_path  - path to target directory.
#     args         - additional arguments for tar.
unpack_archive() {
    archive_path="$1"
    target_path="$2"
    shift 2
    mkdir -p "$target_path"
    tar xfz "$archive_path" -C "$target_path" "$@"
}


# Installs Java SDK.
# Usage: <resource_id> <java_home> <temp_archive>
#     resource_id  - Java SDK resource ID to download.
#     java_home    - target Java home directory.
#     temp_archive - temporary file to download archive into.
install_java_sdk() {
    resource_id="$1"
    java_home="$2"
    temp_archive="$3"

    # Download an install Java SDK.

    download_sandbox_resource "$resource_id" "$temp_archive"
    unpack_archive "$temp_archive" "$java_home" --strip-components=1
    rm "$temp_archive"

    # Update alternatives.

    update-alternatives --install "/usr/bin/java" "java" "$java_home/bin/java" 1
    update-alternatives --install "/usr/bin/javac" "javac" "$java_home/bin/javac" 1
    update-alternatives --set java "$java_home/bin/java"
    update-alternatives --set javac "$java_home/bin/javac"

    # Install Yandex certificate.

    CERT_URL="https://crls.yandex.net/YandexInternalRootCA.crt"
    CERT_FILE="/usr/local/share/ca-certificates/YandexInternalRootCA.crt"
    wget "$CERT_URL" -O "$CERT_FILE"
    update-ca-certificates
    "$java_home/bin/keytool" -importcert -file "$CERT_FILE" \
        -keystore "$java_home/jre/lib/security/cacerts" \
        -alias Yandex -storepass changeit -noprompt

    # Update environment variables.

    echo "export JAVA_HOME=$java_home"        >> /etc/profile.d/jdk.sh
    echo "export JAVA8_HOME=$java_home"       >> /etc/profile.d/jdk.sh
    echo "export PATH=\$PATH:\$JAVA_HOME/bin" >> /etc/profile.d/jdk.sh
}


# Installs Apache Maven.
# Usage: <resource_id> <maven_home> <temp_archive>
#     resource_id  - Apache Maven resource ID to download.
#     maven_home   - target Maven home directory.
#     temp_archive - temporary file to download archive into.
install_maven() {
    resource_id="$1"
    maven_home="$2"
    temp_archive="$3"

    # Download an install Apache Maven.

    download_sandbox_resource "$resource_id" "$temp_archive"
    unpack_archive "$temp_archive" "$maven_home" --strip-components=1
    rm "$temp_archive"

    # Update environment variables.

    echo "export M2_HOME=$maven_home"       >> /etc/profile.d/maven.sh
    echo "export PATH=\$PATH:\$M2_HOME/bin" >> /etc/profile.d/maven.sh
}
