"""
Copyright (c) 2017 Yandex LLC. All rights reserved.
Author: Vladimir Stefanovskiy <stef@yandex-team.ru>
"""
import logging
import subprocess
import os

from sandbox import sdk2
import sandbox.common.types.client as ctc
from sandbox.sandboxsdk.environments import PipEnvironment


INFRA_BOARD_REPO_URL = (
    'https://bitbucket.browser.yandex-team.ru/scm/~stef/infra-board.git')

INFRA_BOARD_DIR = 'infra-board'

PYTHON_REQUIREMENTS = (
    os.path.join(INFRA_BOARD_DIR, 'requirements.txt'),
    os.path.join(INFRA_BOARD_DIR, 'deps/yin/requirements.txt'),
)
VIRTUALENV = 'check_infra_health'


class BrowserMobileCheckInfraHealth(sdk2.Task):

    class Parameters(sdk2.Parameters):
        _container = sdk2.parameters.Container(
            "Container", default=None, required=True)

        yav_token_vault = sdk2.parameters.String(
            'Vault item with yav token', default='robot-browser-infra_yav_token')

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.BROWSER & ctc.Tag.LINUX_TRUSTY
        environments = (
            PipEnvironment('virtualenv', '15.1.0'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    def _prepare_virtualenv(self):
        subprocess.check_call(['virtualenv', VIRTUALENV])
        binpath = os.path.abspath(os.path.join(VIRTUALENV, 'bin'))
        os.environ['PATH'] = binpath + os.pathsep + os.environ['PATH']
        for req_file in PYTHON_REQUIREMENTS:
            subprocess.check_call(['pip', 'install', '-r', req_file,
                                   '--verbose', '-U', '-I'])

    def on_execute(self):
        logging.info('Cloning infra-board repository')
        subprocess.check_call(['git', 'clone', INFRA_BOARD_REPO_URL,
                              INFRA_BOARD_DIR])

        logging.info('Installing deps')
        subprocess.check_call(['bash', 'install_deps.sh'],
                              cwd=INFRA_BOARD_DIR)

        logging.info('Installing python requirements')
        self._prepare_virtualenv()

        env = os.environ.copy()
        env['YAV_TOKEN'] = sdk2.Vault.data(self.Parameters.yav_token_vault)
        logging.info('Starting runner script')
        subprocess.check_call(['python', 'run_checkers.py'],
                              cwd=INFRA_BOARD_DIR, env=env)
