#!/usr/bin/env python
"""
Author: Vladimir Stefanovskiy <stef@yandex-team.ru>

Utility to evict old ccache records from mongodb.
"""
import argparse
import logging
from itertools import islice

from pymongo import MongoClient, DeleteOne, ASCENDING
from pymongo.errors import CursorNotFound
from humanize import naturalsize


def collection_size(collection):
    stats = collection.database.command('collstats', collection.name)
    return stats['size']


def list_oldest_ids(collection, cursor_batch=50):

    def get_ids_batch():
        batch = []
        try:
            for doc in collection.find(
                    projection=[],
                    sort=[('touch', ASCENDING)],
                    limit=cursor_batch):
                assert 'value' not in doc
                batch.append(doc['_id'])
        except CursorNotFound:
            pass
        return batch

    while True:
        ids = get_ids_batch()
        if not ids:
            return
        for _id in ids:
            yield _id


def evict_records(args):
    logging.info('Connecting to %s', args.mongo_uri)
    client = MongoClient(args.mongo_uri)
    db = client.get_default_database()
    cache = db.cache
    old_ids_gen = list_oldest_ids(cache)

    while True:
        if collection_size(cache) <= args.target_size:
            logging.info('Target cache size is reached.')
            break
        logging.info(
            'Current cache size is %s (bigger than target size %s).',
            naturalsize(collection_size(cache), binary=True),
            naturalsize(args.target_size, binary=True))
        ids = islice(old_ids_gen, args.drop_batch)
        deletes = [DeleteOne(filter={'_id': _id}) for _id in ids]
        if not deletes:
            logging.warning('No more records to delete')
            break
        logging.info('Dropping %d records', len(deletes))
        cache.bulk_write(deletes, ordered=False)

    logging.info('Current cache size is %s',
                 naturalsize(collection_size(cache), binary=True))


def main():
    parser = argparse.ArgumentParser(
        description='Utility to evict old ccache records from mongodb.')
    parser.add_argument('-v', '--verbose', action='store_true',
                        help='Enable debug logging')
    parser.add_argument('--mongo-uri', metavar='URI', required=True,
                        help='Mongodb URI')
    parser.add_argument('--target-size', metavar='BYTES', type=int,
                        required=True,
                        help='Keep droppind old records while cache size '
                             'is bigger than target size.')
    parser.add_argument('--drop-batch', metavar='COUNT', type=int, default=100,
                        help='Number of records to drop in one iteration.')
    args = parser.parse_args()

    logging.basicConfig(format='[%(levelname)s] %(message)s',
                        level=logging.DEBUG if args.verbose else logging.INFO)

    evict_records(args)


if __name__ == '__main__':
    main()
