# -*- coding: utf-8 -*-
from sandbox import sdk2
import json
import requests

PLATFORM_API = "https://platform.yandex-team.ru/api/v1"
DOCKER_INFO = "https://dockinfo.yandex-team.ru/api/docker/resolve?registryUrl={0}&tag={1}"
BOT_URL = 'https://cat.offline.common.yandex.net'


class BsOutdoorToQloud(sdk2.Task):
    """Deploy new docker image to qloud"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        object_id = sdk2.parameters.String("object_id",
                                           default="perspective-products.raskraska.prod", required=True)
        revision = sdk2.parameters.String("revision", default="", required=False)
        component_name = sdk2.parameters.String("component_name", default="raskraska", required=True)
        repository = sdk2.parameters.String("repository",
                                            default="registry.yandex.net/yabs/outdoor/raskraska", required=True)
        commit_author = sdk2.parameters.String("commit_author", default="", required=False)
        commit_message = sdk2.parameters.String("commit_message", default="", required=False)
        db_name = sdk2.parameters.String("db_name", default="", required=False)

    def on_execute(self):
        platform_auth = sdk2.Vault.data('BSOUTDOOR', 'QLOUD_TOKEN')
        docker_auth = sdk2.Vault.data('BSOUTDOOR', 'DOCKER_TOKEN')

        try:
            (registry_url, registry_tag) = self.Parameters.repository.split(":")
        except Exception as e:
            self.set_info("<b>Incorrect registry</b>: {0}".format(self.Parameters.repository), do_escape=False)
            raise e

        req = requests.get(DOCKER_INFO.format(registry_url, registry_tag), headers={
            "Authorization": "OAuth {0}".format(docker_auth)}, verify=False)

        if req.status_code != 200:
            self.set_info("<b>Failed to fetch hash</b>: {0} {1}".format(req.status_code, req.text), do_escape=False)
            raise Exception("Could not fetch hash for image")

        hashId = req.json()["hash"]
        self.set_info("Found hash id {0}".format(hashId), do_escape=False)

        post_data = [{"componentName": self.Parameters.component_name, "properties": {
            "repository": self.Parameters.repository, "hash": hashId}}]
        version_url = "{0}/environment/status/{1}".format(PLATFORM_API, self.Parameters.object_id)

        req = requests.get(version_url, headers={"Authorization": "OAuth {0}".format(platform_auth)}, verify=False)

        if req.status_code != 200:
            self.set_info("<b>Error getting last version</b>: {0} {1}".format(
                req.status_code, req.text), do_escape=False)
            raise Exception("Could not update service")

        last_version = req.json()["version"]
        self.set_info("<b>Last version</b>: {0}".format(last_version), do_escape=False)
        post_url = "{0}/environment/fast-deploy/{1}/{2}".format(PLATFORM_API, self.Parameters.object_id, last_version)

        req = requests.post(post_url, data=json.dumps(post_data), headers={
            "Authorization": "OAuth {0}".format(platform_auth),
            "Content-type": "application/json"}, verify=False)

        if req.status_code != 204:
            self.set_info("<b>Error updating service</b>: {0} {1}".format(req.status_code, req.text), do_escape=False)
            raise Exception("Could not update service")

        self.set_info("<b>Update completed</b>", do_escape=False)

        try:
            response = requests.get(BOT_URL, verify=False, params={
                'action': 'sendMessage',
                'text': '{}.{} deployed https://a.yandex-team.ru/commit/{}\n{}{}'.format(
                    self.Parameters.db_name,
                    self.Parameters.component_name,
                    self.Parameters.revision,
                    self.Parameters.commit_message,
                    ('\nby ' + self.Parameters.commit_author) if self.Parameters.commit_author else '',
                ),
            })
            self.set_info("<b>Cat hooked, status</b>: {}".format(response.status_code), do_escape=False)
        except Exception as exc:
            self.set_info("<b>Bot info failed</b>: {0}".format(exc), do_escape=False)
