# -*- coding: utf-8 -*-

import json
import os
import time

import sandbox.common.types.resource as ctr
import sandbox.projects.resource_types as rt

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.bsyeti.common import (
    basic_releasers, LongAbExperimentsConfig, LongestAbExperimentsConfig
)
from sandbox.projects.common import solomon


class MakeBigbAbExperimentsConfigBin(rt.ARCADIA_PROJECT):
    """
    resource with binary of ads/bsyeti/scripts/make_bigb_ab_experiments_config
    """

    releasable = True
    any_arch = True
    auto_backup = True
    releasers = basic_releasers
    release_subscribers = basic_releasers


class BigbAbExperimentsProductionConfig(sdk2.Resource):
    """
    Bigb config of ab experiments (Production). Contains only bigb (144) configuration from experiment storage
    """
    any_arch = True
    auto_backup = True
    calc_md5 = True
    share = True
    releasable = True
    releasers = basic_releasers


class BigbAbExperimentsPreproductionConfig(sdk2.Resource):
    """
    Bigb config of ab experiments (Preproduction). Contains only bigb (144) configuration from experiment storage
    """
    any_arch = True
    auto_backup = True
    calc_md5 = True
    share = True
    releasable = True
    releasers = basic_releasers


class BigbAbExperimentsTestingConfig(sdk2.Resource):
    """
    Bigb config of ab experiments (Testing). Contains only bigb (144) configuration from experiment storage
    """
    any_arch = True
    auto_backup = True
    calc_md5 = True
    share = True
    releasable = True
    releasers = basic_releasers


class MakeBigbAbExperimentsConfig2(sdk2.Task):
    """
    task to build bigb ab experiments config
    """
    class Caches(sdk2.Requirements.Caches):
        pass  # means that task do not use any shared caches

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 20000

    class Parameters(sdk2.Parameters):
        binary_id = sdk2.parameters.LastReleasedResource(
            'resource with binary of ads/bsyeti/scripts/make_bigb_ab_experiments_config',
            resource_type=MakeBigbAbExperimentsConfigBin,
            state=(ctr.State.READY, ),
            required=True
        )

        add_test_id_tree_info = sdk2.parameters.Bool(
            "Fetch test id tree info from adminka",
            default=False,
            required=False,
        )

        add_halting_points = sdk2.parameters.Bool(
            "Add list of testids turned off with ITS",
            default=False,
            required=False,
        )

        long_config = sdk2.parameters.Bool(
            "Do build LONG config",
            default=False,
            required=False,
            do_not_copy=True
        )
        longest_config = sdk2.parameters.Bool(
            "Do build LONGEST config (another resource type, we want to store more days than LONG config)",
            default=False,
            required=False,
            do_not_copy=True
        )
        history_days_count = sdk2.parameters.Integer(
            "Load history configs of AB experiments for last <history_days_count> days",
            default=0,
            required=True,
        )

        with sdk2.parameters.RadioGroup('Tag') as ab_tag:
            ab_tag.values['production'] = ab_tag.Value(value='production', default=True)
            ab_tag.values['preproduction'] = ab_tag.Value(value='preproduction')
            ab_tag.values['testing'] = ab_tag.Value(value='testing')

        config_name = sdk2.parameters.String(
            "Name of file with config in output resource",
            default="long_ab_config.json",
            required=True,
        )

        take_production_long_config = sdk2.parameters.Bool(
            "Take production long config",
            default=False,
            required=False,
        )

        ttl_days = sdk2.parameters.Integer(
            "TTL (days) of output resource",
            default=14,
            required=True,
        )

        split_config_to_chunks = sdk2.parameters.Integer(
            "Split config into chunks when writing to table",
            default=1,
            required=True
        )

        sandbox_blacklist = sdk2.parameters.String(
            "Space-separated list of ids of sandbox resources which will be excluded from long config",
            default="",
            required=False,
        )

        yt_clusters = sdk2.parameters.String(
            "Comma-separated list of YT clusters to upload config",
            default="",
            required=True,
        )

        yt_path = sdk2.parameters.String(
            "Path on YT clusters for uploading config",
            default="",
            required=True,
        )

        solomon_tag = sdk2.parameters.String(
            "Tag for solomon metrics",
            default="",
            required=True,
        )

        do_release = sdk2.parameters.Bool(
            "Make this resource released (USE ONLY FOR SCHEDULERS!)",
            default=False,
            required=False,
            do_not_copy=True
        )

        secret_for_robot_owner = sdk2.parameters.String(
            "Secret for robot owner",
            default="sec-01djab78jh9dvsymnjzwhk1mpf",  # zomb-yeti token
            required=True,
        )

        host_for_push_to_solomon = sdk2.parameters.String(
            "Host for pushing event to solomon",
            default="default-host",
            required=False,
        )

        with sdk2.parameters.Output:
            message = sdk2.parameters.String("Message", multiline=True)

        upload_to_table = sdk2.parameters.Bool(
            "Also upload to table",
            default=False,
            required=False,
            do_not_copy=True
        )

    def on_execute(self):
        bin_res = sdk2.ResourceData(self.Parameters.binary_id)

        counters_file = "counters.json"

        cmd = [
            str(bin_res.path),
            "--days-count", str(self.Parameters.history_days_count),
            "--tag", self.Parameters.ab_tag,
            "--ab-long-config-name", self.Parameters.config_name,
            "--yt-upload-clusters", self.Parameters.yt_clusters,
            "--yt-upload-path", self.Parameters.yt_path,
            "--counters-file", counters_file,
            "--split-config-to-chunks", str(self.Parameters.split_config_to_chunks),
        ]
        env = dict(os.environ)
        env["YT_TOKEN"] = sdk2.Vault.data("{}[yt_token]".format(self.Parameters.secret_for_robot_owner))  # robot token
        env["AB_OAUTH_TOKEN"] = sdk2.Vault.data("{}[ab_token]".format(self.Parameters.secret_for_robot_owner))  # robot token

        if self.Parameters.sandbox_blacklist:
            cmd += ["--sandbox-blacklist"] + self.Parameters.sandbox_blacklist.split()

        if self.Parameters.add_test_id_tree_info:
            cmd += ["--add-test-id-tree-info"]

        if self.Parameters.add_halting_points:
            cmd += ["--add-halting-points"]
            env["NANNY_TOKEN"] = sdk2.Vault.data("{}[nanny_token]".format(self.Parameters.secret_for_robot_owner))  # robot token

        if self.Parameters.take_production_long_config:
            cmd += ["--take-production-long-config"]

        if self.Parameters.upload_to_table:
            cmd += ["--upload-to-table"]

        with sdk2.helpers.ProcessLog(self, logger="bigb_make_bigb_ab_experiments_config") as l:
            sp.check_call(cmd, env=env, stdout=l.stdout, stderr=l.stderr)
            with open(str(l.stdout.path)) as f:
                self.Parameters.message = f.read()

        if self.Parameters.longest_config:
            experimentsConfigType = LongestAbExperimentsConfig
        elif self.Parameters.long_config:
            experimentsConfigType = LongAbExperimentsConfig
        else:
            experimentsConfigType = {
                "production": BigbAbExperimentsProductionConfig,
                "preproduction": BigbAbExperimentsPreproductionConfig,
                "testing": BigbAbExperimentsTestingConfig,
            }[self.Parameters.ab_tag]

        res = experimentsConfigType(
            self,
            "Bigb ab experiments config (%s)" % self.Parameters.ab_tag,
            self.Parameters.config_name,
            ttl=self.Parameters.ttl_days,
        )
        if self.Parameters.do_release:
            res.released = 'stable'

        with open(counters_file) as f:
            counters = json.load(f)
        counters["success"] = 1
        current_time = int(time.time())

        solomon.push_to_solomon_v2(
            token=sdk2.Vault.data("{}[solomon_token]".format(self.Parameters.secret_for_robot_owner)),  # robot token
            params={
                'project': 'bigb_oracle',
                'cluster': 'sandbox_tasks',
                'service': 'sandbox_tasks',
            },
            common_labels=(('host', self.Parameters.host_for_push_to_solomon),),
            sensors=[
                {

                    'labels': {
                        'task': 'MAKE_BIGB_AB_EXPERIMENTS_CONFIG',
                        'tag': self.Parameters.solomon_tag,
                        'sensor': sensor,
                    },
                    'ts': current_time,
                    'value': value,
                }
                for sensor, value in counters.items()
            ],
        )
