# -*- coding: utf-8 -*-

import os
import time

import sandbox.common.types.resource as ctr
import sandbox.projects.resource_types as rt

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.bsyeti.common import basic_releasers
from sandbox.sandboxsdk.channel import channel


class BigbFlamegraphBaseBuilder(rt.ARCADIA_PROJECT):
    """
    resource with binary building flamegraph base
    """

    releasable = True
    any_arch = True
    auto_backup = True
    releasers = basic_releasers
    release_subscribers = basic_releasers


class BigbFlamegraphBase(sdk2.Resource):
    """
    Base with cpu profiling results, allowing to draw flamegraphs
    """

    any_arch = True
    auto_backup = True
    calc_md5 = True
    share = True
    releasable = True
    releasers = basic_releasers
    producer = sdk2.parameters.String("BigB")


class MakeBigbFlamegraphBase(sdk2.Task):
    """
    task to build geo dict with permalinks
    """

    class Caches(sdk2.Requirements.Caches):
        pass  # means that task do not use any shared caches

    class Requirements(sdk2.Requirements):
        cores = 2  # exactly 2 cores
        ram = 10000

    class Parameters(sdk2.Parameters):
        builder_id = sdk2.parameters.LastReleasedResource(
            "builder",
            resource_type=BigbFlamegraphBaseBuilder,
            state=(ctr.State.READY,),
            required=True,
        )
        producer = sdk2.parameters.String(
            "Name of actor, that produces profiler logs",
            default="bigb",
            required=False,
        )
        vault = sdk2.parameters.String(
            "vault yav token, with yt_token and yql_token",
            default="sec-01djab78jh9dvsymnjzwhk1mpf",
            required=False,
        )
        mail_to = sdk2.parameters.String(
            "List of mails, comma separated",
            default="",
            required=False,
        )
        logfeller_paths = sdk2.parameters.List(
            'list of paths for logfeller queues',
            default=[
                "//logs/bigb-poormansprofiler-test-log/1d",
                "//logs/bigb-poormansprofiler-production-log/1d",
                "//logs/adv-machine-poormansprofiler-log/1d",
            ],
            required=False,
        )
        namespaces = sdk2.parameters.List(
            'list of names for resultings bases',
            default=["1 WITH 2"],
            required=False,
        )
        exps = sdk2.parameters.List(
            'list of experiments allowed in bases',
            default=["apphost", "default"],
            required=False,
        )
        do_release = sdk2.parameters.Bool(
            "Make this resourse released (USE ONLY FOR SCHEDULERS!)",
            default=False,
            required=False,
        )

    def on_execute(self):
        log = sdk2.helpers.ProcessLog(self, logger="bigb_profiler_loggin")
        builder_res = sdk2.ResourceData(self.Parameters.builder_id)

        folder = "flames"
        os.mkdir(folder)
        vault = self.Parameters.vault
        env = {
            "YT_TOKEN": sdk2.Vault.data(vault + "[yt_token]"),
            "YQL_TOKEN": sdk2.Vault.data(vault + "[yql_token]"),
            "TMP": ".",
            "TMPDIR": ".",
        }

        t = time.time()

        cmd_args = []
        cmd_args.append(str(builder_res.path))

        for e in self.Parameters.exps:
            cmd_args.append("--exp")
            cmd_args.append(str(e))

        for ns in self.Parameters.namespaces:
            cmd_args.append("--namespace")
            cmd_args.append(str(ns))

        for path in self.Parameters.logfeller_paths:
            cmd_args.append("--table")
            cmd_args.append(str(path))

        cmd_args.append("--base")
        cmd_args.append("profiler_base.pb")

        sp.check_call(
            cmd_args, stdout=log.stdout, stderr=log.stderr, env=env
        )

        res = BigbFlamegraphBase(
            self, "BigbB Profiler Results", "profiler_base.pb", producer=self.Parameters.producer
        )
        log.close()
        if self.Parameters.do_release:
            res.released = "stable"
        if self.Parameters.mail_to != "":
            channel.sandbox.send_email(
                self.Parameters.mail_to.split(","),
                [],
                "Bigb poor mans profiler results from {}".format(time.ctime(int(t))),
                "Link to resource with results: https://proxy.sandbox.yandex-team.ru/{}".format(
                    res.id
                ),
            )
