import datetime
import os
import time

import pytz

import sandbox.common.types.resource
import sandbox.projects.resource_types as resource_types

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess
from sandbox.projects.bsyeti.common import (
    dump_timestamp,
    basic_releasers,
    BigbFlamegraphPlotterBin,
)
from sandbox.sandboxsdk.channel import channel


def get_profile_size_info_file_name(cluster, table_path, field_name=None):
    if field_name:
        return "{}{}_{}".format(
            cluster,
            table_path.replace("/", "_"),
            field_name,
        )
    else:
        return "{}{}".format(
            cluster,
            table_path.replace("/", "_"),
        )


class LocalExecutor(object):
    def __init__(
        self,
        profile_size_analyzer_resource_path,
        flamegraph_plotter_resource_path,
        current_dt,
        yt_pool,
        env,
        logger,
    ):
        self._profile_size_analyzer_resource_path = str(
            profile_size_analyzer_resource_path
        )
        self._flamegraph_plotter_resource_path = str(flamegraph_plotter_resource_path)
        self._logger = logger
        self._env = env
        self._yt_pool = yt_pool
        self._current_dt = current_dt
        self._current_dt_str = current_dt.strftime("%Y-%m-%dT%H-%M")
        self._output_folder_for_plots = "plots"
        os.mkdir(self._output_folder_for_plots)

        self._current_profile_sizes_folder = "sizes_{}".format(self._current_dt_str)
        os.mkdir(self._current_profile_sizes_folder)

    def fetch_old_profile_sizes_resource(self, age_in_days):
        dt_start = self._current_dt - datetime.timedelta(days=age_in_days)

        profile_size_info_resources = [
            resource
            for resource in BigbProfileSizesInfo.find(attrs={"released": "stable"}, state="READY").order(-sdk2.Resource.id).limit(100)
            if dt_start >= resource.created
        ]

        if not profile_size_info_resources:
            return None, None

        oldest_profile_sizes_info_resource = profile_size_info_resources[0]

        return sdk2.ResourceData(oldest_profile_sizes_info_resource), oldest_profile_sizes_info_resource.created

    def calculate_profile_sizes(
        self, path_to_table, cluster, profile_name, field_name=None, sample_rate=0.01,
    ):
        fetcher_cmd = [
            self._profile_size_analyzer_resource_path,
            "--path",
            path_to_table,
            "--cluster",
            cluster,
            "--sample-rate",
            str(sample_rate),
            "--profile",
            profile_name,
            "--pool",
            self._yt_pool,
            "--mode",
            "weights",
        ]
        if field_name is not None:
            fetcher_cmd += ["--field", str(field_name)]

        output_file_name = os.path.join(
            self._current_profile_sizes_folder,
            get_profile_size_info_file_name(cluster, path_to_table, field_name),
        )

        with open(output_file_name, "w") as output_file:
            subprocess.check_call(
                fetcher_cmd,
                stdout=output_file,
                stderr=self._logger.stderr,
                env=self._env,
            )

        return output_file_name

    def draw_flame(self, profile_size_info_file_path):
        output_file_path = os.path.join(
            self._output_folder_for_plots,
            "{}_{}_flamegraph.html".format(
                os.path.splitext(os.path.basename(profile_size_info_file_path))[0],
                self._current_dt_str,
            ),
        )
        flames_cmd = [
            str(self._flamegraph_plotter_resource_path),
            "--in-data",
            profile_size_info_file_path,
            "--out-svg",
            output_file_path,
        ]
        subprocess.check_call(
            flames_cmd,
            stdout=self._logger.stdout,
            stderr=self._logger.stderr,
            env=self._env,
        )

    def draw_compare_flame(
        self, old_profile_sizes_folder, old_dt, cluster, path_to_table, field_name
    ):

        old_profile_sizes_info_file_path = os.path.join(
            old_profile_sizes_folder,
            get_profile_size_info_file_name(cluster, path_to_table, field_name),
        )

        if not os.path.exists(old_profile_sizes_info_file_path):
            return

        current_profile_sizes_info_file_path = os.path.join(
            self._current_profile_sizes_folder,
            get_profile_size_info_file_name(cluster, path_to_table, field_name),
        )

        flames_cmd = [
            str(self._flamegraph_plotter_resource_path),
            "--in-data",
            old_profile_sizes_info_file_path,
            "--compare",
            current_profile_sizes_info_file_path,
            "--out-svg",
            os.path.join(
                self._output_folder_for_plots,
                "{}_compared_with_{}.html".format(
                    "{}_{}".format(
                        os.path.splitext(
                            os.path.basename(old_profile_sizes_info_file_path)
                        )[0],
                        old_dt.strftime("%Y-%m-%dT%H-%M"),
                    ),
                    "{}_{}".format(
                        os.path.splitext(
                            os.path.basename(current_profile_sizes_info_file_path)
                        )[0],
                        self._current_dt_str,
                    ),
                ),
            ),
        ]

        subprocess.check_call(
            flames_cmd,
            stdout=self._logger.stdout,
            stderr=self._logger.stderr,
            env=self._env,
        )

    def publish_results(self, task, do_release, mail_to):
        timestamp = time.mktime(self._current_dt.timetuple())
        dump_timestamp(self._output_folder_for_plots, timestamp)
        result = BigbProfileSizePlotterResults(
            task,
            "Bigb Profile Size Plotter Results",
            self._output_folder_for_plots,
        )
        new_profile_sizes = BigbProfileSizesInfo(
            task,
            "Bigb Profile Size Info",
            self._current_profile_sizes_folder,
        )
        new_profile_sizes.released = "stable"
        self._logger.close()

        if do_release:
            result.released = "stable"
        if mail_to:
            channel.sandbox.send_email(
                mail_to,
                [],
                "Bigb profile size analyzer results from {}".format(
                    time.ctime(int(timestamp))
                ),
                "Link to resource with results: https://proxy.sandbox.yandex-team.ru/{}".format(
                    result.id
                ),
            )


class BigbProfileSizeAnalyzerBin(resource_types.ARCADIA_PROJECT):
    """
    Resource with fetcher preparing profile size results.
    """

    releasable = True
    any_arch = True
    auto_backup = True
    releasers = basic_releasers
    release_subscribers = basic_releasers


class BigbProfileSizePlotterResults(sdk2.Resource):
    """
    Result flame graphs of profile sizes.
    """

    any_arch = True
    auto_backup = True
    calc_md5 = True
    share = True
    releasable = True
    releasers = basic_releasers


class BigbProfileSizesInfo(sdk2.Resource):
    """
    Profile sizes in DSV format.
    """

    any_arch = True
    auto_backup = True
    calc_md5 = True
    share = True
    releasable = True
    releasers = basic_releasers


class MakeBigbProfileSizeFlames(sdk2.Task):
    """
    Task to build flamegraphs with profile size information.
    """

    class Caches(sdk2.Requirements.Caches):
        pass  # means that task do not use any shared caches

    class Requirements(sdk2.Requirements):
        cores = 2  # exactly 2 cores
        ram = 10000

    class Parameters(sdk2.Parameters):
        profile_size_analyzer_resource_id = sdk2.parameters.LastReleasedResource(
            "Bigb profile size analyzer",
            resource_type=BigbProfileSizeAnalyzerBin,
            state=(sandbox.common.types.resource.State.READY,),
            required=True,
        )
        flamegraph_plotter_resource_id = sdk2.parameters.LastReleasedResource(
            "Flames",
            resource_type=BigbFlamegraphPlotterBin,
            state=(sandbox.common.types.resource.State.READY,),
            required=True,
        )
        clusters = sdk2.parameters.List(
            "List of clusters",
            default=["hahn"],
            required=False,
        )
        paths = sdk2.parameters.List(
            "List of paths to logs",
            default=[
                "//home/bigb/caesar/public_dumps/stable/{}".format(profile)
                for profile in [
                    "AdGroups",
                    "Banners",
                    "DspCreatives",
                    "Offers",
                    "Orders",
                    "Pages",
                ]
            ],
            required=False,
        )
        profile_names = sdk2.parameters.List(
            "List of profile names in the order corresponding to the list of paths",
            default=[
                "AdGroupProfile",
                "BannerProfile",
                "DspCreativeProfile",
                "OfferProfile",
                "OrderProfile",
                "PageProfile",
            ],
            required=False,
        )
        field_names = sdk2.parameters.List(
            "List of field names in the order corresponding to the list of paths. Use this if profile is in the field.",
            default=None,
            required=False,
        )
        sample_rates = sdk2.parameters.List(
            "List of sample rates for profile size analyzer in the order corresponding to the list of paths. May be empty for default sample rates.",
            default=[],
            required=False,
        )
        yt_pool = sdk2.parameters.String(
            "Yt pool for profile size analyzer",
            default="bigb",
            required=False,
        )
        vault = sdk2.parameters.String(
            "Vault yav token, with yt_token and yql_token",
            default="sec-01djab78jh9dvsymnjzwhk1mpf",
            required=False,
        )
        mail_to = sdk2.parameters.List(
            "List of mails",
            default=[],
            required=False,
        )
        age_in_days = sdk2.parameters.Integer(
            "The youngest resource with information about the weight of the profiles "
            "whose age in days will be more than this parameter will be selected for comparison.",
            default=7,
            required=False,
        )
        do_release = sdk2.parameters.Bool(
            "Make this resourse released (USE ONLY FOR SCHEDULERS!)",
            default=False,
            required=False,
        )

    def on_execute(self):
        if len(self.Parameters.profile_names) != len(self.Parameters.paths):
            raise ValueError("Length of profile_names and paths should be equal.")
        if len(self.Parameters.field_names) != 0 and len(self.Parameters.paths) != len(
            self.Parameters.field_names
        ):
            raise ValueError(
                "Length of field_names should be 0 or be equal with paths' length."
            )
        if len(self.Parameters.sample_rates) != 0 and len(self.Parameters.sample_rates) != len(
            self.Parameters.paths
        ):
            raise ValueError(
                "Length of sample_rates should be 0 or be equal with paths' length."
            )

        logger = sdk2.helpers.ProcessLog(self, logger="bigb_profiler_loggin")
        profile_size_analyzer_resource = sdk2.ResourceData(
            self.Parameters.profile_size_analyzer_resource_id
        )
        flamegraph_plotter_resource = sdk2.ResourceData(
            self.Parameters.flamegraph_plotter_resource_id
        )

        vault = self.Parameters.vault
        env = {
            "YT_TOKEN": sdk2.Vault.data(vault + "[yt_token]"),
            "YQL_TOKEN": sdk2.Vault.data(vault + "[yql_token]"),
            "TMP": ".",
            "TMPDIR": ".",
        }

        executor = LocalExecutor(
            profile_size_analyzer_resource.path,
            flamegraph_plotter_resource.path,
            datetime.datetime.utcnow().replace(tzinfo=pytz.UTC),
            self.Parameters.yt_pool,
            env,
            logger,
        )

        old_profile_sizes_resource, old_profile_size_datetime = executor.fetch_old_profile_sizes_resource(
            self.Parameters.age_in_days
        )

        for cluster in self.Parameters.clusters:
            for i in range(len(self.Parameters.paths)):
                path_to_table = self.Parameters.paths[i]
                profile_name = self.Parameters.profile_names[i]
                field_name = self.Parameters.field_names[i] if self.Parameters.field_names else None
                sample_rate = self.Parameters.sample_rates[i] if self.Parameters.sample_rates else 0.01

                profile_size_info_file_path = executor.calculate_profile_sizes(
                    path_to_table, cluster, profile_name, field_name, sample_rate
                )
                executor.draw_flame(profile_size_info_file_path)

                if old_profile_sizes_resource is not None:
                    executor.draw_compare_flame(
                        str(old_profile_sizes_resource.path),
                        old_profile_size_datetime,
                        cluster,
                        path_to_table,
                        field_name,
                    )

        executor.publish_results(
            self,
            self.Parameters.do_release,
            self.Parameters.mail_to,
        )
